package mysql

import (
	"context"
	"fmt"
	"strconv"

	"github.com/jmoiron/sqlx"

	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/filter"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/model"
)

const selectFallbacksQueryTemplate = "SELECT id, srcgate, srcname, dstgate, dstname, `order`, event_create, audit_bulk.timestamp AS event_create_ts, event_modify, audit_bulk2.timestamp AS event_modify_ts FROM fallbacks %s WHERE %s ORDER BY id LIMIT ?"
const selectFallbacksPredicate = "id > ?"

const deleteFallbacksQuery = "DELETE FROM fallbacks WHERE id IN (:delete)"

const updateFallbacksQuery = "UPDATE fallbacks SET srcgate = :srcgate, srcname = :srcname, dstgate = :dstgate, dstname = :dstname, `order` = :order  WHERE id = :id"

const insertFallbacksQuery = "INSERT INTO fallbacks (srcgate, srcname, dstgate, dstname, `order`) VALUES (:srcgate, :srcname, :dstgate, :dstname, :order)"

const fallbacksTableName = "fallbacks"

type FallbacksQueryParams struct {
	ID      uint64 `db:"id"`
	SrcGate string `db:"srcgate"`
	SrcName string `db:"srcname"`
	DstGate string `db:"dstgate"`
	DstName string `db:"dstname"`
	Order   int16  `db:"order"`
}

var fallbacksFilterFields = map[string]string{
	model.FallBackIDAlias: "id",
	model.SrcgateAlias:    "srcgate",
	model.SrcnameAlias:    "srcname",
	model.DstgateAlias:    "dstgate",
	model.DstnameAlias:    "dstname",
	model.OrderAlias:      "`order`",
}

func (provider *Provider) GetFallbacksCount(ctx context.Context, fallbacksFilter filter.Filter) (uint64, error) {
	return provider.GetCount(ctx, "fallbacks", fallbacksFilter, fallbacksFilterFields)
}

func (provider *Provider) GetFallbacks(
	ctx context.Context,
	fromID model.EntityID,
	limit uint64,
	filter filter.Filter) ([]*model.Fallback, error) {
	return GetEntityInfo[model.Fallback, GetFallbacksResult](ctx, &limitSelectArgs{fromID, limit}, filter, DBEntitySpec{
		filterFields:        fallbacksFilterFields,
		selectPredicate:     selectFallbacksPredicate,
		selectQueryTemplate: selectFallbacksQueryTemplate,
		entityLogTitle:      "fallbacks",
		tableName:           fallbacksTableName,
	}, provider)
}

func (provider *Provider) SetFallbacks(
	ctx context.Context,
	delete []model.EntityID,
	create []*model.Fallback,
	update []*model.Fallback,
	auditLogBulkParams *model.AuditLogBulkParams,
) error {
	batch := make(queryBatch, 0, 1+len(update)+len(create))

	/* NOTE: https://a.yandex-team.ru/arc_vcs/passport/infra/daemons/yasms_internal/internal/model/mysql/routes.go?rev=r8947415#L108 */
	if err := prepareDeleteQuery(&batch, deleteFallbacksQuery, fallbacksTableName, delete, Delete); err != nil {
		return fmt.Errorf("failed to prepare delete fallbacks query: %s", err)
	}

	if err := prepareFallbacksQueries(&batch, updateFallbacksQuery, update, Update); err != nil {
		return fmt.Errorf("failed to prepare delete update query: %s", err)
	}

	if err := prepareFallbacksQueries(&batch, insertFallbacksQuery, create, Insert); err != nil {
		return fmt.Errorf("failed to prepare delete create query: %s", err)
	}

	_, err := provider.withDriver(func(db *sqlx.DB) (interface{}, error) {
		return nil, batch.execSingleTransactionWithAuditLog(db, ctx, auditLogBulkParams)
	})
	return err
}

func prepareFallbacksQueries(batch *queryBatch, queryTemplate string, update []*model.Fallback, setType SetType) error {
	for _, fallback := range update {
		var id uint64
		var insertedID *model.EntityID
		var err error
		if fallback.ID == "" {
			insertedID = &fallback.ID
		} else {
			if id, err = strconv.ParseUint(fallback.ID, 10, 64); err != nil {
				return err
			}
		}

		query, args, err := prepareNamedQuery(
			queryTemplate,
			&FallbacksQueryParams{
				ID:      id,
				SrcGate: fallback.SrcGate,
				SrcName: fallback.SrcName,
				DstGate: fallback.DstGate,
				DstName: fallback.DstName,
				Order:   fallback.Order,
			})
		if err != nil {
			return err
		}

		batch.append(&queryHolder{
			Query:        query,
			Args:         args,
			AffectedRows: 1,
			InsertedID:   insertedID,
			QueryMetaData: queryMetaData{
				ID:        []model.EntityID{fallback.ID},
				TableName: fallbacksTableName,
				Type:      setType,
				Payload:   fallback.String(),
			},
		})
	}

	return nil
}
