package mysql

import (
	"context"
	"fmt"
	"strconv"

	"github.com/jmoiron/sqlx"

	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/filter"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/model"
)

const selectGatesQueryTemplate = `SELECT gateid, aliase, fromname, description, event_create, audit_bulk.timestamp AS event_create_ts, event_modify, audit_bulk2.timestamp AS event_modify_ts
FROM smsgates %s
WHERE %s ORDER BY gateid LIMIT ?;`
const selectAllGatesQueryTemplate = `SELECT gateid, aliase, fromname, description FROM smsgates %s ORDER BY gateid;`
const selectGatesPredicate = "gateid > ?"

const deleteGatesQuery = `
DELETE
FROM smsgates
WHERE smsgates.gateid IN (:delete)
  AND smsgates.gateid NOT IN (SELECT DISTINCT *
                              from (SELECT gateid AS gate
                                    FROM smsrt
                                    UNION ALL
                                    SELECT gateid2 AS gate
                                    FROM smsrt
                                    UNION ALL
                                    SELECT gateid3 AS gate
                                    FROM smsrt) AS gates
                              WHERE gates.gate != 0)
`

const insertGatesQuery = `INSERT INTO smsgates (aliase, fromname, description) VALUES (:aliase, :fromname, :description)`

const updateGatesQuery = `UPDATE smsgates SET aliase=:aliase, fromname=:fromname, description=:description WHERE gateid=:gateid`

const gatesTableName = "smsgates"

type GatesQueryParams struct {
	ID          uint64 `db:"gateid"`
	Alias       string `db:"aliase"`
	AlphaName   string `db:"fromname"`
	Description string `db:"description"`
}

var gatesFilterFields = map[string]string{
	model.GateIDFieldAlias:      "gateid",
	model.AliaseFieldAlias:      "aliase",
	model.FromnameFieldAlias:    "fromname",
	model.DescriptionFieldAlias: "description",
	model.DelayFieldAlias:       "delay",
}

func (provider *Provider) GetGatesCount(ctx context.Context, gatesFilter filter.Filter) (uint64, error) {
	return provider.GetCount(ctx, "smsgates", gatesFilter, gatesFilterFields)
}

func (provider *Provider) GetGates(
	ctx context.Context,
	fromID model.EntityID,
	limit uint64,
	filter filter.Filter) ([]*model.GateWithAudit, error) {
	return GetEntityInfo[model.GateWithAudit, GetGatesResult](ctx, &limitSelectArgs{fromID, limit}, filter, DBEntitySpec{
		filterFields:        gatesFilterFields,
		selectPredicate:     selectGatesPredicate,
		selectQueryTemplate: selectGatesQueryTemplate,
		entityLogTitle:      "gates",
		tableName:           gatesTableName,
	}, provider)
}

func (provider *Provider) GetAllGates(
	ctx context.Context,
) ([]*model.GateWithAudit, error) {
	return GetEntityInfo[model.GateWithAudit, GetGatesResult](ctx, nil, nil, DBEntitySpec{
		selectQueryTemplate: selectAllGatesQueryTemplate,
		entityLogTitle:      "all gates",
	}, provider)
}

func (provider *Provider) SetGates(
	ctx context.Context,
	delete []model.EntityID,
	create []*model.Gate,
	update []*model.Gate,
	auditLogBulkParams *model.AuditLogBulkParams,
) error {
	batch := make(queryBatch, 0, 1+len(update)+len(create))

	/* NOTE: https://a.yandex-team.ru/arc_vcs/passport/infra/daemons/yasms_internal/internal/model/mysql/routes.go?rev=r8947415#L108 */
	if err := prepareDeleteQuery(&batch, deleteGatesQuery, gatesTableName, delete, Delete); err != nil {
		return fmt.Errorf("failed to prepare delete gates query: %s", err)
	}

	if err := prepareGateQueries(&batch, updateGatesQuery, update, Update); err != nil {
		return fmt.Errorf("failed to prepare update gates query: %s", err)
	}

	if err := prepareGateQueries(&batch, insertGatesQuery, create, Insert); err != nil {
		return fmt.Errorf("failed to prepare create gates query: %s", err)
	}

	_, err := provider.withDriver(func(db *sqlx.DB) (interface{}, error) {
		return nil, batch.execSingleTransactionWithAuditLog(db, ctx, auditLogBulkParams)
	})
	return err
}

func prepareGateQueries(batch *queryBatch, queryTemplate string, update []*model.Gate, setType SetType) error {
	for _, gate := range update {
		var id uint64
		var insertedID *model.EntityID
		var err error
		if gate.ID == "" {
			insertedID = &gate.ID
		} else {
			if id, err = strconv.ParseUint(gate.ID, 10, 64); err != nil {
				return err
			}
		}

		query, args, err := prepareNamedQuery(
			queryTemplate,
			&GatesQueryParams{
				ID:          id,
				Alias:       gate.Alias,
				AlphaName:   gate.AlphaName,
				Description: packGateDescription(gate),
			})
		if err != nil {
			return err
		}

		batch.append(&queryHolder{
			Query:        query,
			Args:         args,
			AffectedRows: 1,
			InsertedID:   insertedID,
			QueryMetaData: queryMetaData{
				ID:        []model.EntityID{gate.ID},
				TableName: gatesTableName,
				Type:      setType,
				Payload:   gate.String(),
			},
		})
	}

	return nil
}
