package mysql

import (
	"context"
	"fmt"

	"github.com/jmoiron/sqlx"

	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/filter"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/model"
)

const eventInfoTemplate = `
LEFT JOIN audit_row ON audit_row.row_id=%[1]s.event_create
LEFT JOIN audit_row AS audit_row2 ON audit_row2.row_id=%[1]s.event_modify
LEFT JOIN audit_bulk ON audit_row.bulk_id=audit_bulk.bulk_id
LEFT JOIN audit_bulk AS audit_bulk2 ON audit_row2.bulk_id=audit_bulk2.bulk_id
`

type DBEntitySpec struct {
	filterFields        map[string]string
	selectQueryTemplate string
	entityLogTitle      string
	selectPredicate     string
	tableName           string
}

type EntityResultInterface interface {
	GetID() int64
	MakeEntity() (model.EntityInfoInterface, error)
}

func GetEntityInfo[EntityInfo model.RouteInfo | model.GateWithAudit | model.Fallback | model.BlockedPhone | model.Template,
	GetResult GetRoutesResult | GetGatesResult | GetFallbacksResult | GetBlockedPhonesResult | GetTemplatesResult](
	ctx context.Context,
	selectArgs *limitSelectArgs,
	filter filter.Filter,
	entitySpec DBEntitySpec,
	provider *Provider) ([]*EntityInfo, error) {

	var results []GetResult
	query, args, err := PrepareSelectInfoQuery(selectArgs, filter, entitySpec)
	if err != nil {
		return nil, fmt.Errorf("failed to prepare select %s query: %s", entitySpec.entityLogTitle, err)
	}

	_, err = provider.withDriver(func(db *sqlx.DB) (interface{}, error) {
		return nil, db.SelectContext(ctx, &results, db.Rebind(query), args...)
	})
	if err != nil {
		return nil, fmt.Errorf("failed to select %s: %s, %s, %s", entitySpec.entityLogTitle, err, query, args)
	}

	entities := make([]*EntityInfo, 0, len(results))
	for _, result := range results {
		entity, err := any(result).(EntityResultInterface).MakeEntity()
		if err != nil {
			return nil, fmt.Errorf("failed to make %s for id=%d: %s", entitySpec.entityLogTitle, any(result).(EntityResultInterface).GetID(), err)
		}
		entities = append(entities, entity.(*EntityInfo))
	}
	return entities, nil
}
