package mysql

import (
	"context"
	"database/sql"
	"fmt"
	"strconv"
	"strings"

	"github.com/jmoiron/sqlx"

	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/filter"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/model"
)

type AuditInfo struct {
	EventCreateID sql.NullString `db:"event_create"`
	EventCreateTS sql.NullInt64  `db:"event_create_ts"`
	EventModifyID sql.NullString `db:"event_modify"`
	EventModifyTS sql.NullInt64  `db:"event_modify_ts"`
}

type GetRoutesResult struct {
	EntityResultInterface
	RuleID      sql.NullInt64  `db:"ruleid"`
	PhonePrefix sql.NullString `db:"destination"`
	Weight      sql.NullInt16  `db:"weight"`
	Mode        sql.NullString `db:"mode"`

	GateID1      sql.NullInt64  `db:"gateid1"`
	Alias1       sql.NullString `db:"aliase1"`
	AlphaName1   sql.NullString `db:"fromname1"`
	Description1 sql.NullString `db:"description1"`

	GateID2      sql.NullInt64  `db:"gateid2"`
	Alias2       sql.NullString `db:"aliase2"`
	AlphaName2   sql.NullString `db:"fromname2"`
	Description2 sql.NullString `db:"description2"`

	GateID3      sql.NullInt64  `db:"gateid3"`
	Alias3       sql.NullString `db:"aliase3"`
	AlphaName3   sql.NullString `db:"fromname3"`
	Description3 sql.NullString `db:"description3"`

	AuditInfo
}

func (result GetRoutesResult) MakeEntity() (model.EntityInfoInterface, error) {
	if !result.Mode.Valid {
		return nil, fmt.Errorf("mode is empty")
	}
	gates, err := result.makeGates()
	if err != nil {
		return nil, err
	}

	return &model.RouteInfo{
		ID:          strconv.FormatInt(result.RuleID.Int64, 10),
		PhonePrefix: result.PhonePrefix.String,
		Weight:      result.Weight.Int16,
		Gates:       gates,
		Mode:        result.Mode.String,
		AuditCreate: model.EventInfo{ChangeID: result.EventCreateID.String, TS: result.EventCreateTS.Int64},
		AuditModify: model.EventInfo{ChangeID: result.EventModifyID.String, TS: result.EventModifyTS.Int64},
	}, nil
}

func (result GetRoutesResult) GetID() int64 {
	return result.RuleID.Int64
}

func unpackGateDescription(gate *model.Gate, description string) error {
	parts := strings.Split(description, ";")
	var extra map[string]string
	for _, part := range parts {
		if part == "" {
			continue
		}

		keyValue := strings.Split(part, "=")
		if len(keyValue) != 2 {
			return fmt.Errorf("invalid key-value pair: %s", part)
		}

		if keyValue[0] == "contractor" {
			gate.Contractor = keyValue[1]
			continue
		}

		if keyValue[0] == "consumer" {
			gate.Consumer = keyValue[1]
			continue
		}

		if extra == nil {
			extra = map[string]string{}
		}
		extra[keyValue[0]] = keyValue[1]
	}
	gate.Extra = extra
	return nil
}

func packGateDescription(gate *model.Gate) string {
	parts := []string{
		fmt.Sprintf("consumer=%s;contractor=%s;", gate.Consumer, gate.Contractor),
	}

	for key, value := range gate.Extra {
		parts = append(parts, fmt.Sprintf("%s=%s;", key, value))
	}

	return strings.Join(parts, "")
}

func addGate(gates *[]*model.Gate, id sql.NullInt64, alias sql.NullString, alphaName sql.NullString, description sql.NullString) error {
	if !id.Valid || !alias.Valid || !alphaName.Valid {
		return nil
	}
	gate := &model.Gate{
		ID:        strconv.FormatInt(id.Int64, 10),
		Alias:     alias.String,
		AlphaName: alphaName.String,
	}
	if description.Valid {
		err := unpackGateDescription(gate, description.String)
		if err != nil {
			return err
		}
	}

	*gates = append(*gates, gate)
	return nil
}

func (result *GetRoutesResult) makeGates() ([]*model.Gate, error) {
	if !result.GateID1.Valid {
		return nil, fmt.Errorf("")
	}
	gates := make([]*model.Gate, 0, 3)
	err := addGate(&gates, result.GateID1, result.Alias1, result.AlphaName1, result.Description1)
	if err != nil {
		return nil, err
	}
	err = addGate(&gates, result.GateID2, result.Alias2, result.AlphaName2, result.Description2)
	if err != nil {
		return nil, err
	}
	err = addGate(&gates, result.GateID3, result.Alias3, result.AlphaName3, result.Description3)
	if err != nil {
		return nil, err
	}
	return gates, nil
}

func getGateIDAt(route *model.Route, i int) model.EntityID {
	if i >= len(route.Gates) {
		return "0"
	}

	return route.Gates[i]
}

type GetGatesResult struct {
	EntityResultInterface
	GateID      sql.NullInt64  `db:"gateid"`
	Alias       sql.NullString `db:"aliase"`
	AlphaName   sql.NullString `db:"fromname"`
	Description sql.NullString `db:"description"`

	AuditInfo
}

func (result GetGatesResult) MakeEntity() (model.EntityInfoInterface, error) {
	gate := &model.GateWithAudit{
		Gate: model.Gate{
			ID:        strconv.FormatInt(result.GateID.Int64, 10),
			Alias:     result.Alias.String,
			AlphaName: result.AlphaName.String,
		},
		EntityCommon: model.EntityCommon{
			AuditCreate: model.EventInfo{ChangeID: result.EventCreateID.String, TS: result.EventCreateTS.Int64},
			AuditModify: model.EventInfo{ChangeID: result.EventModifyID.String, TS: result.EventModifyTS.Int64},
		},
	}
	err := unpackGateDescription(&gate.Gate, result.Description.String)
	if err != nil {
		return nil, err
	}
	return gate, nil
}

func (result GetGatesResult) GetID() int64 {
	return result.GateID.Int64
}

type GetBlockedPhonesResult struct {
	EntityResultInterface
	ID          sql.NullInt64  `db:"blockid"`
	PhoneNumber sql.NullString `db:"phone"`
	BlockType   sql.NullString `db:"blocktype"`
	BlockUntil  sql.NullTime   `db:"blocktill"`

	AuditInfo
}

func (result GetBlockedPhonesResult) MakeEntity() (model.EntityInfoInterface, error) {
	if !result.BlockType.Valid {
		return nil, fmt.Errorf("empty block type")
	}
	blockType := model.BlockType(result.BlockType.String)

	return &model.BlockedPhone{
		ID:          strconv.FormatInt(result.ID.Int64, 10),
		PhoneNumber: result.PhoneNumber.String,
		BlockType:   blockType,
		BlockUntil:  result.BlockUntil.Time,
		AuditCreate: model.EventInfo{ChangeID: result.EventCreateID.String, TS: result.EventCreateTS.Int64},
		AuditModify: model.EventInfo{ChangeID: result.EventModifyID.String, TS: result.EventModifyTS.Int64},
	}, nil
}

func (result GetBlockedPhonesResult) GetID() int64 {
	return result.ID.Int64
}

type GetFallbacksResult struct {
	EntityResultInterface
	ID      sql.NullInt64  `db:"id"`
	SrcGate sql.NullString `db:"srcgate"`
	SrcName sql.NullString `db:"srcname"`
	DstGate sql.NullString `db:"dstgate"`
	DstName sql.NullString `db:"dstname"`
	Order   sql.NullInt16  `db:"order"`

	AuditInfo
}

func (result GetFallbacksResult) MakeEntity() (model.EntityInfoInterface, error) {
	return &model.Fallback{
		ID:          strconv.FormatInt(result.ID.Int64, 10),
		SrcGate:     result.SrcGate.String,
		SrcName:     result.SrcName.String,
		DstGate:     result.DstGate.String,
		DstName:     result.DstName.String,
		Order:       result.Order.Int16,
		AuditCreate: model.EventInfo{ChangeID: result.EventCreateID.String, TS: result.EventCreateTS.Int64},
		AuditModify: model.EventInfo{ChangeID: result.EventModifyID.String, TS: result.EventModifyTS.Int64},
	}, nil
}

func (result GetFallbacksResult) GetID() int64 {
	return result.ID.Int64
}

func prepareNamedQuery(queryTemplate string, data interface{}) (string, []interface{}, error) {
	query, args, err := sqlx.Named(queryTemplate, data)
	if err != nil {
		return "", nil, err
	}
	return sqlx.In(query, args...)
}

const selectCountQueryTemplate = `
SELECT COUNT(*) as count
FROM %s
%s`

func prepareSelectCountQuery(
	tableName string,
	filter filter.Filter,
	filterFields map[string]string,
) (string, []interface{}, error) {
	var args []interface{}
	var predicate string
	if filter != nil {
		filterQuery, filterArgs, err := filter.ToMySQLQuery(filterFields, args)
		if err != nil {
			return "", nil, err
		}

		predicate, args, err = sqlx.In(fmt.Sprintf("WHERE %s\n", filterQuery), filterArgs...)
		if err != nil {
			return "", nil, err
		}
	}

	return fmt.Sprintf(selectCountQueryTemplate, tableName, predicate), args, nil
}

func (provider *Provider) GetCount(
	ctx context.Context,
	fromSubst string,
	filter filter.Filter,
	filterFields map[string]string,
) (uint64, error) {
	query, args, err := prepareSelectCountQuery(fromSubst, filter, filterFields)
	if err != nil {
		return 0, fmt.Errorf("failed to prepare select from %s count query: %s", fromSubst, err)
	}

	var count uint64
	_, err = provider.withDriver(func(db *sqlx.DB) (interface{}, error) {
		return nil, db.QueryRowContext(ctx, db.Rebind(query), args...).Scan(&count)
	})
	if err != nil {
		return 0, fmt.Errorf("failed to select from %s count: %s", fromSubst, err)
	}

	return count, nil
}

type limitSelectArgs struct {
	fromID model.EntityID
	limit  uint64
}

func PrepareSelectInfoQuery(selectArgs *limitSelectArgs, fieldsFilter filter.Filter, entitySpec DBEntitySpec) (string, []interface{}, error) {
	args := make([]interface{}, 0)

	if selectArgs != nil {
		id, err := strconv.ParseUint(selectArgs.fromID, 10, 64)
		if err != nil {
			return "", nil, err
		}
		args = append(args, id)
	}

	if fieldsFilter != nil {
		var err error
		var filterQuery string
		filterQuery, args, err = fieldsFilter.ToMySQLQuery(entitySpec.filterFields, args)
		if err != nil {
			return "", nil, err
		}
		entitySpec.selectPredicate = fmt.Sprint(entitySpec.selectPredicate, " AND ", filterQuery)
	}

	if selectArgs != nil {
		args = append(args, selectArgs.limit)
	}

	var query, eventInfoQueryPart string

	if entitySpec.tableName != "" {
		eventInfoQueryPart = fmt.Sprintf(eventInfoTemplate, entitySpec.tableName)
	}

	if entitySpec.selectPredicate != "" {
		query = fmt.Sprintf(entitySpec.selectQueryTemplate, eventInfoQueryPart, entitySpec.selectPredicate)
	} else {
		query = fmt.Sprintf(entitySpec.selectQueryTemplate, eventInfoQueryPart)
	}
	return sqlx.In(query, args...)
}

func PrepareSelectRegionsQuery(fieldsFilter filter.Filter, filterFields map[string]string, sqlTemplate string) (string, []interface{}, error) {
	eventInfoQueryPart := fmt.Sprintf(eventInfoTemplate, regionsTableName)
	if fieldsFilter != nil {
		var err error
		var filterQuery string
		args := make([]interface{}, 0)

		filterQuery, args, err = fieldsFilter.ToMySQLQuery(filterFields, args)

		if err != nil {
			return "", nil, err
		}

		query := fmt.Sprintf(sqlTemplate, eventInfoQueryPart, "WHERE "+filterQuery)
		return sqlx.In(query, args...)
	}
	return fmt.Sprintf(sqlTemplate, eventInfoQueryPart, ""), nil, nil
}

type GetRegionsResult struct {
	EntityResultInterface
	ID     sql.NullInt64  `db:"id"`
	Prefix sql.NullString `db:"prefix"`
	Name   sql.NullString `db:"name"`

	AuditInfo
}

func (result GetRegionsResult) MakeEntity() (*model.Region, error) {
	return &model.Region{
		ID:     strconv.FormatInt(result.ID.Int64, 10),
		Prefix: result.Prefix.String,
		Name:   result.Name.String,
		EntityCommon: model.EntityCommon{
			AuditCreate: model.EventInfo{ChangeID: result.EventCreateID.String, TS: result.EventCreateTS.Int64},
			AuditModify: model.EventInfo{ChangeID: result.EventModifyID.String, TS: result.EventModifyTS.Int64},
		},
	}, nil
}

func (result GetRegionsResult) GetID() int64 {
	return result.ID.Int64
}

type SetType int

const (
	Insert SetType = iota
	Update
	Delete
)

func (setType SetType) String() string {
	switch setType {
	case Insert:
		return "add"
	case Delete:
		return "delete"
	case Update:
		return "update"
	}
	return "unknown"
}

func (setType SetType) NeedUpdateData() bool {
	return setType == Insert || setType == Update
}

type GetAuditInfoResultBase struct {
	ID      sql.NullInt64  `db:"bulk_id"`
	Author  sql.NullString `db:"author"`
	Issue   sql.NullString `db:"issue"`
	Comment sql.NullString `db:"comment"`
	TS      sql.NullInt64  `db:"timestamp"`

	RowID    sql.NullInt64  `db:"row_id"`
	Type     sql.NullString `db:"type"`
	EntityID sql.NullInt64  `db:"entity_id"`
	Payload  sql.NullString `db:"payload"`
}

type GetAuditBulkInfoResultWithPayload struct {
	GetAuditInfoResultBase
	Payload sql.NullString `db:"payload"`
}

type GetTemplatesResult struct {
	EntityResultInterface
	ID                sql.NullInt64  `db:"id"`
	Text              sql.NullString `db:"text"`
	AbcService        sql.NullString `db:"abc_service"`
	SenderMeta        sql.NullString `db:"sender_meta"`
	FieldsDescription sql.NullString `db:"fields_description"`

	AuditInfo
}

func (result GetTemplatesResult) MakeEntity() (model.EntityInfoInterface, error) {
	return &model.Template{
		ID:                strconv.FormatInt(result.ID.Int64, 10),
		Text:              result.Text.String,
		AbcService:        result.AbcService.String,
		SenderMeta:        result.SenderMeta.String,
		FieldsDescription: result.FieldsDescription.String,
		AuditCreate:       model.EventInfo{ChangeID: result.EventCreateID.String, TS: result.EventCreateTS.Int64},
		AuditModify:       model.EventInfo{ChangeID: result.EventModifyID.String, TS: result.EventModifyTS.Int64},
	}, nil
}

func (result GetTemplatesResult) GetID() int64 {
	return result.ID.Int64
}
