package mysql

import (
	"fmt"
	"io/ioutil"
	"strings"
	"sync"

	"github.com/go-sql-driver/mysql"
	"github.com/jmoiron/sqlx"

	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/errs"
	"a.yandex-team.ru/passport/shared/golibs/juggler"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

type ProviderConfig struct {
	Host string `json:"host"`
	Port uint16 `json:"port"`

	UsernameFile string `json:"user"`
	PasswordFile string `json:"password"`

	DBName string `json:"db"`

	Lazy bool `json:"lazy"`
}

type Provider struct {
	driver *sqlx.DB
	config *mysql.Config
	mutex  sync.RWMutex
}

type mysqlJob func(*sqlx.DB) (interface{}, error)

func NewMySQLProvider(params ProviderConfig) (*Provider, error) {
	if params.DBName == "" {
		return nil, &errs.RuntimeError{
			Message:   "Database is not specified",
			Component: errs.ProviderComponent,
		}
	}

	config := mysql.NewConfig()

	// TODO: if on unix, use socket connection
	config.Net = "tcp"
	config.Addr = fmt.Sprintf("%s:%d", params.Host, params.Port)

	user, err := ioutil.ReadFile(params.UsernameFile)
	if err != nil {
		return nil, err
	}

	password, err := ioutil.ReadFile(params.PasswordFile)
	if err != nil {
		return nil, err
	}

	config.User = strings.TrimSpace(string(user))
	config.Passwd = strings.TrimSpace(string(password))

	config.DBName = params.DBName
	config.ParseTime = true

	driver, err := sqlx.Connect("mysql", config.FormatDSN())
	if err != nil {
		if !params.Lazy {
			return nil, err
		}

		logger.Log().Errorf("cannot initialize mysql: %s, skipping for now", err.Error())
	}

	return &Provider{
		driver: driver,
		config: config,
	}, nil
}

func (provider *Provider) GetName() string {
	return fmt.Sprint("mysql:", provider.config.DBName)
}

func (provider *Provider) isInitialized() bool {
	provider.mutex.RLock()
	defer provider.mutex.RUnlock()
	return provider.driver != nil
}

func (provider *Provider) initDriver() error {
	provider.mutex.Lock()
	defer provider.mutex.Unlock()

	if provider.driver != nil {
		return nil
	}

	var err error
	provider.driver, err = sqlx.Connect("mysql", provider.config.FormatDSN())
	if err != nil {
		return err
	}

	return nil
}

func (provider *Provider) withDriver(callback mysqlJob) (interface{}, error) {
	if !provider.isInitialized() {
		// NOTE: we are making an assumption here that provider can be initialized but cannot be destroyed
		err := provider.initDriver()
		if err != nil {
			return nil, err
		}
	}

	provider.mutex.RLock()
	defer provider.mutex.RUnlock()
	return callback(provider.driver)
}

func (provider *Provider) Ping() error {
	_, err := provider.withDriver(func(db *sqlx.DB) (interface{}, error) {
		return nil, db.Ping()
	})

	return err
}

func (provider *Provider) GetJugglerStatus() *juggler.Status {
	if err := provider.Ping(); err != nil {
		return juggler.NewStatus(juggler.Critical, "Mysql unavailable: %s", err)
	}
	return juggler.NewStatusOk()
}
