package mysql

import (
	"context"
	"fmt"
	"strconv"

	"github.com/jmoiron/sqlx"

	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/filter"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/model"
)

const selectAllRegionsQueryTemplate = "SELECT id, prefix, name, event_create, audit_bulk.timestamp AS event_create_ts, event_modify, audit_bulk2.timestamp AS event_modify_ts FROM regions %s %s ORDER BY id"

const deleteRegionsQuery = "DELETE FROM regions WHERE id IN (:delete)"

const updateRegionsQuery = "UPDATE regions SET name = :name  WHERE id = :id"

const insertRegionsQuery = "INSERT INTO regions (prefix, name) VALUES (:prefix, :name)"

const regionsTableName = "regions"

type RegionsQueryParams struct {
	ID     uint64 `db:"id"`
	Prefix string `db:"prefix"`
	Name   string `db:"name"`
}

var regionsFilterFields = map[string]string{
	model.PrefixFieldAlias: "prefix",
	model.NameFieldAlias:   "name",
}

func (provider *Provider) GetRegions(
	ctx context.Context,
	regionsFilter filter.Filter,
) ([]*model.Region, error) {
	var results []GetRegionsResult

	query, args, err := PrepareSelectRegionsQuery(regionsFilter, regionsFilterFields, selectAllRegionsQueryTemplate)
	if err != nil {
		return nil, fmt.Errorf("failed to prepare select regions info query: %s", err)
	}

	_, err = provider.withDriver(func(db *sqlx.DB) (interface{}, error) {
		return nil, db.SelectContext(ctx, &results, db.Rebind(query), args...)
	})
	if err != nil {
		return nil, fmt.Errorf("failed to select regions: %s %s %s", err, query, args)
	}

	regions := make([]*model.Region, 0, len(results))
	for _, result := range results {
		region, err := result.MakeEntity()
		if err != nil {
			return nil, fmt.Errorf("failed to make regions for id=%d: %s", result.ID.Int64, err)
		}
		regions = append(regions, region)
	}
	return regions, nil
}

func (provider *Provider) SetRegions(
	ctx context.Context,
	delete []model.EntityID,
	create []*model.Region,
	update []*model.Region,
	auditLogBulkParams *model.AuditLogBulkParams,
) error {
	batch := make(queryBatch, 0, 1+len(update)+len(create))

	if err := prepareDeleteQuery(&batch, deleteRegionsQuery, regionsTableName, delete, Delete); err != nil {
		return fmt.Errorf("failed to prepare regions delete query: %s", err)
	}

	if err := prepareRegionsQueries(&batch, updateRegionsQuery, update, Update); err != nil {
		return fmt.Errorf("failed to prepare regions update query: %s", err)
	}

	if err := prepareRegionsQueries(&batch, insertRegionsQuery, create, Insert); err != nil {
		return fmt.Errorf("failed to prepare regions create query: %s", err)
	}

	_, err := provider.withDriver(func(db *sqlx.DB) (interface{}, error) {
		return nil, batch.execSingleTransactionWithAuditLog(db, ctx, auditLogBulkParams)
	})
	return err
}

func prepareRegionsQueries(batch *queryBatch, queryTemplate string, update []*model.Region, setType SetType) error {
	for _, region := range update {
		var id uint64
		var insertedID *model.EntityID
		var err error
		if region.ID == "" {
			insertedID = &region.ID
		} else {
			if id, err = strconv.ParseUint(region.ID, 10, 64); err != nil {
				return err
			}
		}
		query, args, err := prepareNamedQuery(
			queryTemplate,
			&RegionsQueryParams{
				ID:     id,
				Prefix: region.Prefix,
				Name:   region.Name,
			})
		if err != nil {
			return err
		}

		batch.append(&queryHolder{
			Query:        query,
			Args:         args,
			AffectedRows: 1,
			InsertedID:   insertedID,
			QueryMetaData: queryMetaData{
				ID:        []model.EntityID{region.ID},
				TableName: regionsTableName,
				Type:      setType,
				Payload:   region.String(),
			},
		})
	}

	return nil
}
