package mysql

import (
	"context"
	"fmt"
	"strconv"

	"github.com/jmoiron/sqlx"

	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/filter"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/model"
)

const selectRoutesInfoQueryTemplate = `
SELECT routes.ruleid AS ruleid,
       routes.destination AS destination,
       routes.weight AS weight,
       routes.mode AS mode,
       gates.gateid AS gateid1,
       gates.aliase AS aliase1,
       gates.fromname AS fromname1,
       gates.description AS description1,
       gates2.gateid AS gateid2,
       gates2.aliase AS aliase2,
       gates2.fromname AS fromname2,
       gates2.description AS description2,
       gates3.gateid AS gateid3,
       gates3.aliase AS aliase3,
       gates3.fromname AS fromname3,
       gates3.description AS description3,
       routes.event_create as event_create, audit_bulk.timestamp AS event_create_ts, routes.event_modify AS event_modify, audit_bulk2.timestamp AS event_modify_ts
FROM
smsrt AS routes
LEFT OUTER JOIN smsgates AS gates ON (routes.gateid = gates.gateid)
LEFT OUTER JOIN smsgates AS gates2 ON (routes.gateid2 = gates2.gateid)
LEFT OUTER JOIN smsgates AS gates3 ON (routes.gateid3 = gates3.gateid)
%s
WHERE %s
ORDER BY routes.ruleid
LIMIT ?
`
const selectRoutesPredicate = "ruleid > ?"
const selectCountRoutesInfo = `smsrt AS routes
	LEFT OUTER JOIN smsgates AS gates ON (routes.gateid = gates.gateid)
    LEFT OUTER JOIN smsgates AS gates2 ON (routes.gateid2 = gates2.gateid)
    LEFT OUTER JOIN smsgates AS gates3 ON (routes.gateid3 = gates3.gateid)
`

const routesTableName = "smsrt"

var routesFilterFields = map[string]string{
	model.GateIDFieldAlias:      "routes.gateid",
	model.GateID2FieldAlias:     "routes.gateid2",
	model.GateID3FieldAlias:     "routes.gateid3",
	model.DestinationFieldAlias: "routes.destination",
	model.ModeFieldAlias:        "routes.mode",
	model.WeightFieldAlias:      "routes.weight",
	model.AliaseFieldAlias:      "gates.aliase",
	model.Aliase2FieldAlias:     "gates2.aliase",
	model.Aliase3FieldAlias:     "gates3.aliase",
	model.FromnameFieldAlias:    "gates.fromname",
	model.Fromname2FieldAlias:   "gates2.fromname",
	model.Fromname3FieldAlias:   "gates3.fromname",
	model.NameFieldAlias:        "regions.name",
}

const deleteRoutesQuery = `DELETE FROM smsrt WHERE ruleid IN (:delete)`

const insertRoutesQuery = `
INSERT INTO smsrt (destination, weight, mode, gateid, gateid2, gateid3)
SELECT :destination, :weight, :mode,
       (SELECT max(gateid) AS gateid FROM smsgates WHERE gateid = :gate1),
       (SELECT if(:gate2=0, 0, max(gateid)) AS gateid FROM smsgates WHERE gateid = :gate2),
       (SELECT if(:gate3=0, 0, max(gateid)) AS gateid FROM smsgates WHERE gateid = :gate3);
`

const updateRoutesQuery = `
UPDATE smsrt AS routes,
    (SELECT max(gateid) AS gateid FROM smsgates WHERE gateid = :gate1) AS gate1,
    (SELECT if(:gate2=0, 0, max(gateid)) AS gateid FROM smsgates WHERE gateid = :gate2) AS gate2,
    (SELECT if(:gate3=0, 0, max(gateid)) AS gateid FROM smsgates WHERE gateid = :gate3) AS gate3
SET routes.destination=:destination, routes.weight=:weight, routes.mode=:mode,
    routes.gateid     = gate1.gateid,
    routes.gateid2    = gate2.gateid,
    routes.gateid3    = gate3.gateid
WHERE ruleid = :id;
`

type RoutesQueryParams struct {
	ID          uint64          `db:"id"`
	Destination string          `db:"destination"`
	Weight      int16           `db:"weight"`
	Mode        model.RouteMode `db:"mode"`
	GateID1     model.EntityID  `db:"gate1"`
	GateID2     model.EntityID  `db:"gate2"`
	GateID3     model.EntityID  `db:"gate3"`
}

func (provider *Provider) GetRoutesCount(ctx context.Context, routesFilter filter.Filter) (uint64, error) {
	return provider.GetCount(ctx, selectCountRoutesInfo, routesFilter, routesFilterFields)
}

func (provider *Provider) GetRoutesInfo(
	ctx context.Context,
	fromID model.EntityID,
	limit uint64,
	filter filter.Filter) ([]*model.RouteInfo, error) {
	return GetEntityInfo[model.RouteInfo, GetRoutesResult](ctx, &limitSelectArgs{fromID, limit}, filter, DBEntitySpec{
		filterFields:        routesFilterFields,
		selectPredicate:     selectRoutesPredicate,
		selectQueryTemplate: selectRoutesInfoQueryTemplate,
		entityLogTitle:      "routes",
		tableName:           "routes",
	}, provider)
}

func (provider *Provider) SetRoutes(
	ctx context.Context,
	delete []model.EntityID,
	create []*model.Route,
	update []*model.Route,
	auditLogBulkParams *model.AuditLogBulkParams,
) error {
	batch := make(queryBatch, 0, 1+len(update)+len(create))

	/* NOTE: The order is important here, do NOT change it
		- if you try to modify rows that are due to be deleted - there may be an issue, so the patch should not be applied
	    - if you try to modify rows with IDs that do not yet exist - there has been an error, but if we insert first,
			you may create an entry with a matching ID and modify that, which is illegal
	*/
	if err := prepareDeleteQuery(&batch, deleteRoutesQuery, routesTableName, delete, Delete); err != nil {
		return fmt.Errorf("failed to prepare delete routes query: %s", err)
	}

	if err := prepareRouteQueries(&batch, updateRoutesQuery, update, Update); err != nil {
		return fmt.Errorf("failed to prepare update routes query: %s", err)
	}

	if err := prepareRouteQueries(&batch, insertRoutesQuery, create, Insert); err != nil {
		return fmt.Errorf("failed to prepare create routes query: %s", err)
	}

	_, err := provider.withDriver(func(db *sqlx.DB) (interface{}, error) {
		return nil, batch.execSingleTransactionWithAuditLog(db, ctx, auditLogBulkParams)
	})
	return err
}

func prepareRouteQueries(batch *queryBatch, queryTemplate string, update []*model.Route, setType SetType) error {
	for _, route := range update {
		var id uint64
		var insertedID *model.EntityID
		var err error
		if route.ID == "" {
			insertedID = &route.ID
		} else {
			if id, err = strconv.ParseUint(route.ID, 10, 64); err != nil {
				return err
			}
		}

		query, args, err := prepareNamedQuery(
			queryTemplate,
			&RoutesQueryParams{
				ID:          id,
				Destination: route.PhonePrefix,
				Weight:      route.Weight,
				Mode:        route.Mode,
				GateID1:     getGateIDAt(route, 0),
				GateID2:     getGateIDAt(route, 1),
				GateID3:     getGateIDAt(route, 2),
			})
		if err != nil {
			return err
		}

		batch.append(&queryHolder{
			Query:        query,
			Args:         args,
			AffectedRows: 1,
			InsertedID:   insertedID,
			QueryMetaData: queryMetaData{
				ID:        []model.EntityID{route.ID},
				TableName: routesTableName,
				Type:      setType,
				Payload:   route.String(),
			},
		})
	}

	return nil
}
