package mysql

import (
	"context"
	"fmt"
	"strconv"

	"github.com/jmoiron/sqlx"

	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/filter"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/model"
)

const selectTemplatesQueryTemplate = "SELECT id, text, abc_service, sender_meta, fields_description, event_create, audit_bulk.timestamp AS event_create_ts, event_modify, audit_bulk2.timestamp AS event_modify_ts FROM templates %s WHERE %s ORDER BY id LIMIT ?"
const selectTemplatesPredicate = "id > ?"

const updateTemplatesQuery = "UPDATE templates SET abc_service = :abc_service, sender_meta = :sender_meta, fields_description = :fields_description WHERE id = :id"

const insertTemplatesQuery = "INSERT INTO templates (text, abc_service, sender_meta, fields_description) VALUES (:text, :abc_service, :sender_meta, :fields_description)"

const templatesTableName = "templates"

type TemplatesQueryParams struct {
	ID                uint64 `db:"id"`
	Text              string `db:"text"`
	AbcService        string `db:"abc_service"`
	SenderMeta        string `db:"sender_meta"`
	FieldsDescription string `db:"fields_description"`
}

func (provider *Provider) GetTemplatesCount(ctx context.Context, templatesFilter filter.Filter) (uint64, error) {
	return provider.GetCount(ctx, "templates", nil, nil)
}

func (provider *Provider) GetTemplates(
	ctx context.Context,
	fromID model.EntityID,
	limit uint64,
	filter filter.Filter) ([]*model.Template, error) {
	return GetEntityInfo[model.Template, GetTemplatesResult](ctx, &limitSelectArgs{fromID, limit}, filter, DBEntitySpec{
		filterFields:        nil,
		selectPredicate:     selectTemplatesPredicate,
		selectQueryTemplate: selectTemplatesQueryTemplate,
		entityLogTitle:      "templates",
		tableName:           templatesTableName,
	}, provider)
}

func prepareTemplatesQueries(batch *queryBatch, queryTemplate string, update []*model.Template, setType SetType) error {
	for _, template := range update {
		var id uint64
		var insertedID *model.EntityID
		var err error
		if template.ID == "" {
			insertedID = &template.ID
		} else {
			if id, err = strconv.ParseUint(template.ID, 10, 64); err != nil {
				return err
			}
		}

		query, args, err := prepareNamedQuery(
			queryTemplate,
			&TemplatesQueryParams{
				ID:                id,
				Text:              template.Text,
				AbcService:        template.AbcService,
				SenderMeta:        template.SenderMeta,
				FieldsDescription: template.FieldsDescription,
			})
		if err != nil {
			return err
		}

		var payload string
		if setType == Update {
			payload = template.UpdateString()
		} else {
			payload = template.CreateString()
		}

		batch.append(&queryHolder{
			Query:        query,
			Args:         args,
			AffectedRows: 1,
			InsertedID:   insertedID,
			QueryMetaData: queryMetaData{
				ID:        []model.EntityID{template.ID},
				TableName: templatesTableName,
				Type:      setType,
				Payload:   payload,
			},
		})
	}

	return nil
}

func (provider *Provider) SetTemplates(
	ctx context.Context,
	create []*model.Template,
	update []*model.Template,
	auditLogBulkParams *model.AuditLogBulkParams,
) error {
	batch := make(queryBatch, 0, 1+len(update)+len(create))

	if err := prepareTemplatesQueries(&batch, updateTemplatesQuery, update, Update); err != nil {
		return fmt.Errorf("failed to prepare update templates query: %s", err)
	}

	if err := prepareTemplatesQueries(&batch, insertTemplatesQuery, create, Insert); err != nil {
		return fmt.Errorf("failed to prepare create templates query: %s", err)
	}

	_, err := provider.withDriver(func(db *sqlx.DB) (interface{}, error) {
		return nil, batch.execSingleTransactionWithAuditLog(db, ctx, auditLogBulkParams)
	})
	return err
}
