package model

import (
	"encoding/json"
	"fmt"
	"time"
)

const (
	GateIDFieldAlias  = "gateid"
	GateID2FieldAlias = "gateid2"
	GateID3FieldAlias = "gateid3"

	DestinationFieldAlias = "destination"
	ModeFieldAlias        = "mode"
	WeightFieldAlias      = "weight"

	AliaseFieldAlias      = "aliase"
	Aliase2FieldAlias     = "aliase2"
	Aliase3FieldAlias     = "aliase3"
	FromnameFieldAlias    = "fromname"
	Fromname2FieldAlias   = "fromname2"
	Fromname3FieldAlias   = "fromname3"
	DescriptionFieldAlias = "description"
	DelayFieldAlias       = "delay"

	FallBackIDAlias = "id"
	SrcgateAlias    = "srcgate"
	SrcnameAlias    = "srcname"
	DstgateAlias    = "dstgate"
	DstnameAlias    = "dstname"
	OrderAlias      = "order"

	BlockidAlias   = "blockid"
	BlocktillAlias = "blocktill"
	BlocktypeAlias = "blocktype"
	PhoneAlias     = "phone"

	NameFieldAlias   = "name"
	PrefixFieldAlias = "prefix"
)

type BlockType = string

const (
	BlockTypePermanent BlockType = "permanent"
	BlockTypeTemporary BlockType = "temporary"
)

type RouteMode = string

type EntityID = string
type EntityInfoInterface interface {
}

type EntityCommon struct {
	AuditCreate EventInfo `json:"audit_create"`
	AuditModify EventInfo `json:"audit_modify"`
}

type Gate struct {
	ID         EntityID          `json:"gateid"`
	Alias      string            `json:"aliase"` // misspelling for backwards compatibility
	AlphaName  string            `json:"fromname"`
	Consumer   string            `json:"consumer"`
	Contractor string            `json:"contractor"`
	Extra      map[string]string `json:"extra,omitempty"`
}

type GateWithAudit struct {
	Gate
	EntityCommon
}

func (gate *Gate) String() string {
	return fmt.Sprintf("aliase=%s,fromname=%s,consumer=%s,contractor=%s,extra=%s", gate.Alias, gate.AlphaName, gate.Consumer, gate.Contractor, gate.Extra)
}

type Route struct {
	ID          EntityID   `json:"rule_id"`
	PhonePrefix string     `json:"destination"`
	Gates       []EntityID `json:"gates"`
	Weight      int16      `json:"weight"`
	Mode        RouteMode  `json:"mode"`
}

func (route *Route) String() string {
	return fmt.Sprintf("destination=%s,gates=%s,weight=%d,mode=%s", route.PhonePrefix, route.Gates, route.Weight, route.Mode)
}

type RouteInfo struct {
	ID          EntityID  `json:"rule_id"`
	PhonePrefix string    `json:"destination"`
	Gates       []*Gate   `json:"gates"`
	Weight      int16     `json:"weight"`
	Mode        RouteMode `json:"mode"`
	AuditCreate EventInfo `json:"audit_create"`
	AuditModify EventInfo `json:"audit_modify"`
}

type SimpleRegion struct {
	Name string `json:"name"`
}

type SimpleRegionWithAuditInfo struct {
	Name string `json:"name"`
	EntityCommon
}

type RouteFullInfo struct {
	*RouteInfo
	Region *Region `json:"region"`
}

func (routeFullInfo RouteFullInfo) MarshalJSON() ([]byte, error) {
	return json.Marshal(&struct {
		*RouteInfo
		Region SimpleRegion `json:"region"`
	}{
		RouteInfo: routeFullInfo.RouteInfo,
		Region:    SimpleRegion{Name: routeFullInfo.Region.Name},
	})
}

type Enums struct {
	Aliases    map[string][]string `json:"gate_aliase"`
	AlphaNames map[string][]string `json:"fromname"`
}

type BlockedPhone struct {
	ID          EntityID  `json:"blockid"`
	PhoneNumber string    `json:"phone"`
	BlockType   BlockType `json:"blocktype"`
	BlockUntil  time.Time `json:"blocktill"`
	AuditCreate EventInfo `json:"audit_create"`
	AuditModify EventInfo `json:"audit_modify"`
}

func (blockedPhone *BlockedPhone) String() string {
	return fmt.Sprintf("phone=%s,blocktype=%s,blocktill=%s", blockedPhone.PhoneNumber, blockedPhone.BlockType, blockedPhone.BlockUntil)
}

type Fallback struct {
	ID          EntityID  `json:"id"`
	SrcGate     string    `json:"srcgate"`
	SrcName     string    `json:"srcname"`
	DstGate     string    `json:"dstgate"`
	DstName     string    `json:"dstname"`
	Order       int16     `json:"order"`
	AuditCreate EventInfo `json:"audit_create"`
	AuditModify EventInfo `json:"audit_modify"`
}

func (fallback *Fallback) String() string {
	return fmt.Sprintf("srcgate=%s,srcname=%s,dstgate=%s,dstname=%s,order=%d", fallback.SrcGate, fallback.SrcName, fallback.DstName, fallback.DstGate, fallback.Order)
}

type Region struct {
	ID     EntityID `json:"id"`
	Name   string   `json:"name"`
	Prefix string   `json:"prefix"`

	EntityCommon
}

type Regions map[string]*Region

func (region *Region) String() string {
	return fmt.Sprintf("name=%s,prefix=%s", region.Name, region.Prefix)
}

func (regions Regions) CheckValid() error {
	if _, ok := regions["+"]; !ok {
		return fmt.Errorf("no default region provided")
	}
	return nil
}

func (regions Regions) GetRegion(prefix string) (*Region, error) {
	for i := 0; i < len(prefix); i++ {
		if region, ok := regions[prefix[0:len(prefix)-i]]; ok {
			return region, nil
		}
	}
	return nil, fmt.Errorf("region not found in %v", regions)
}

type AuditLogBulkParams struct {
	Author  string   `json:"-"`
	Issue   []string `json:"issue"`
	Comment string   `json:"comment"`
}

type EventInfo struct {
	ChangeID string `json:"change_id"`
	TS       int64  `json:"ts"`
}

type AuditRowChangeBase struct {
	EntityID string `json:"entity_id"`
	Action   string `json:"action"`
}

type AuditRowChange struct {
	AuditRowChangeBase
	Payload string `json:"payload"`
}

type AuditBulkInfoBase struct {
	ID        EntityID `json:"id"`
	Comment   string   `json:"comment"`
	Author    string   `json:"author"`
	Issue     string   `json:"issue"`
	Timestamp int64    `json:"ts"`
}

type AuditBulkInfo struct {
	AuditBulkInfoBase
	Changes map[string]AuditRowChange `json:"changes"`
}

type AuditChangeInfo struct {
	AuditRowChangeBase
	AuditBulkInfo AuditBulkInfoBase `json:"bulk_info"`
}

type AuditChangesInfo map[string]AuditChangeInfo

type Template struct {
	ID                EntityID  `json:"id"`
	AbcService        string    `json:"abc_service"`
	SenderMeta        string    `json:"sender_meta"`
	FieldsDescription string    `json:"fields_description"`
	AuditCreate       EventInfo `json:"audit_create"`
	AuditModify       EventInfo `json:"audit_modify"`
	Text              string    `json:"text"`
}

func (template *Template) UpdateString() string {
	return fmt.Sprintf("abc_service=%s,sender_meta=%s,fields_description=%s", template.AbcService, template.SenderMeta, template.FieldsDescription)
}

func (template *Template) CreateString() string {
	return fmt.Sprintf("text=%s,abc_service=%s,sender_meta=%s,fields_description=%s", template.Text, template.AbcService, template.SenderMeta, template.FieldsDescription)
}
