package ytc

import (
	"context"
	"fmt"

	"a.yandex-team.ru/passport/infra/libs/go/ytsimple"
	"a.yandex-team.ru/passport/shared/golibs/logger"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
)

type rowTypeInterface interface {
	getKey() rowKeysInterface
}

type rowKeysInterface interface {
}

func cleanYtTable[rowType regionRow | gateRow, RowKeysType regionRowKeys | gateRowKeys](
	ctx context.Context,
	queryTemplate string,
	tablePath ypath.Path,
	tx yt.TabletTx,
	provider *Provider,
) error {
	keysToDelete := make([]interface{}, 0)

	query := fmt.Sprintf(
		queryTemplate,
		tablePath.YPath(),
	)

	err := ytsimple.SelectAll(ctx, tx, query, provider.timeout, func(reader yt.TableReader) error {
		var row rowType
		if err := ytsimple.ScanRow(reader, &row); err != nil {
			return err
		}

		keysToDelete = append(keysToDelete, any(row).(rowTypeInterface).getKey())
		return nil
	})

	if err != nil {
		return err
	}

	if err := tx.DeleteRows(ctx, tablePath, keysToDelete, nil); err != nil {
		return fmt.Errorf("failed to delete rows: %w", err)
	}
	return nil
}

func SaveTableToYt[rowType regionRow | gateRow, rowKeysType regionRowKeys | gateRowKeys](
	ctx context.Context,
	rows []interface{},
	queryTemplate string,
	path string,
	provider *Provider,
) error {
	if len(rows) == 0 {
		return nil
	}
	tablePath := buildNodePath(provider.dir, path)
	transaction, err := provider.yc.BeginTabletTx(ctx, nil)
	if err != nil {
		return fmt.Errorf("failed to start transaction: %s", err)
	}
	defer abortTransaction(transaction, err)

	err = cleanYtTable[rowType, rowKeysType](ctx, queryTemplate, tablePath, transaction, provider)
	if err != nil {
		logger.Log().Warnf("YT table %s was not cleaned. Err %s", tablePath, err)
		return err
	}

	err = transaction.InsertRows(ctx, tablePath, rows, nil)
	if err != nil {
		logger.Log().Warnf("YT table %s was not saved: %s. Err %s", tablePath, rows, err)
		return err
	}
	if err = transaction.Commit(); err != nil {
		return fmt.Errorf("failed to commit transaction: %s", err)
	}
	return nil
}
