package ytc

import (
	"context"
	"fmt"
	"strings"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/shared/golibs/juggler"
	"a.yandex-team.ru/passport/shared/golibs/logger"
	"a.yandex-team.ru/passport/shared/golibs/unistat"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yson"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/yt/ythttp"
)

type ProviderConfig struct {
	Cluster string `json:"cluster"`
	Dir     string `json:"dir"`
	Timeout uint64 `json:"timeout_ms"`
}

type Provider struct {
	yc      yt.Client
	timeout yson.Duration
	cluster string
	dir     string
	unistat stats
}

type stats struct {
	responseTimings *unistat.TimeStat
	requests        *unistat.SignalDiff
	errs            *unistat.SignalDiff
}

const TvmYtAlias = "yt"

func NewYtProvider(cfg ProviderConfig, tvmClient tvm.Client) (*Provider, error) {
	yc, err := ythttp.NewClient(&yt.Config{
		Proxy: cfg.Cluster,
		TVMFn: func(ctx context.Context) (string, error) {
			return tvmClient.GetServiceTicketForAlias(ctx, TvmYtAlias)
		},
		Logger:           logger.Log().With(log.Nil("YtClient")).Structured(),
		CompressionCodec: yt.ClientCodecGZIP,
	})
	if err != nil {
		return nil, xerrors.Errorf("Failed to create YT client: %s", err)
	}

	responseTimings, err := unistat.DefaultChunk.CreateTimeStats(
		"yt.response_time",
		unistat.CreateTimeBoundsFromMaxValue(10*time.Second),
	)
	if err != nil {
		return nil, xerrors.Errorf("Failed to create time stats: %w", err)
	}

	return &Provider{
		yc:      yc,
		timeout: yson.Duration(time.Duration(cfg.Timeout) * time.Millisecond),
		cluster: cfg.Cluster,
		dir:     cfg.Dir,
		unistat: stats{
			responseTimings: responseTimings,
			requests:        unistat.DefaultChunk.CreateSignalDiff("yt.requests"),
			errs:            unistat.DefaultChunk.CreateSignalDiff("yt.errors"),
		},
	}, nil
}

func (provider *Provider) GetName() string {
	return fmt.Sprintf("yt:%s:%s", provider.cluster, provider.dir)
}

func (provider *Provider) GetJugglerStatus() *juggler.Status {
	if err := provider.Ping(); err != nil {
		return juggler.NewStatus(juggler.Warning, "YT unavailable: %s", err)
	}
	return juggler.NewStatusOk()
}

func (provider *Provider) Ping() error {
	checkTableExistence := func(path string) error {
		exists, err := provider.yc.NodeExists(context.Background(), buildNodePath(provider.dir, path), nil)
		if err != nil {
			return err
		}
		if !exists {
			return xerrors.Errorf("YT table %s doesn't exist", path)
		}
		return nil
	}
	err := checkTableExistence(regionsTable)
	if err != nil {
		return err
	}
	return checkTableExistence(gatesTable)
}

func (provider *Provider) collectRequestUnistat(start time.Time, err error) {
	provider.unistat.responseTimings.Insert(time.Since(start))
	provider.unistat.requests.Inc()
	if err != nil {
		provider.unistat.errs.Inc()
	}
}

func abortTransaction(tx yt.TabletTx, err error) {
	if err != nil {
		if err := tx.Abort(); err != nil {
			logger.Log().Errorf("failed to abort transaction: %s", err)
		}
	}
}

func buildNodePath(elem ...string) ypath.Path {
	return ypath.Path(strings.Join(elem, "/"))
}
