package roles

import (
	"fmt"

	"a.yandex-team.ru/library/go/yandex/tvm"
)

type AuthTypeSlug string

const (
	AuthWithUserTicket    AuthTypeSlug = "with_user"
	AuthWithoutUserTicket AuthTypeSlug = "without_user"
)

type AccessTypeSlug string

const (
	ReadAccess  AccessTypeSlug = "read"
	WriteAccess AccessTypeSlug = "write"
)

type HandlerSlug string

const (
	BlockedPhonesHandler HandlerSlug = "blockedphones"
	RegionsHandler       HandlerSlug = "regions"
	FallbacksHandler     HandlerSlug = "fallbacks"
	GatesHandler         HandlerSlug = "gates"
	RoutesHandler        HandlerSlug = "routes"
	TemplatesHandler     HandlerSlug = "templates"
	UserRolesHandler     HandlerSlug = "user_roles"
	AuditInfoHandler     HandlerSlug = "audit_info"

	allHandlers HandlerSlug = "all"
)

type UserRolesList map[AccessTypeSlug][]HandlerSlug

var allUserRoles = UserRolesList{
	ReadAccess: {
		BlockedPhonesHandler,
		RegionsHandler,
		FallbacksHandler,
		GatesHandler,
		RoutesHandler,
		TemplatesHandler,
		AuditInfoHandler,
	},
	WriteAccess: {
		BlockedPhonesHandler,
		RegionsHandler,
		FallbacksHandler,
		GatesHandler,
		RoutesHandler,
		TemplatesHandler,
	},
}

func ConvertUserRoles(roles *tvm.ConsumerRoles) UserRolesList {
	result := make(UserRolesList)
	for accessType, handlers := range allUserRoles {
		if roles.HasRole(getUserRoleName(accessType, allHandlers)) {
			result[accessType] = allUserRoles[accessType]
			continue
		}

		result[accessType] = make([]HandlerSlug, 0, 4)
		for _, handler := range handlers {
			if roles.HasRole(getUserRoleName(accessType, handler)) {
				result[accessType] = append(result[accessType], handler)
			}
		}
	}

	return result
}

func CheckServiceRoles(
	roles *tvm.ConsumerRoles,
	authType AuthTypeSlug,
	accessType AccessTypeSlug,
	handle HandlerSlug,
) string {
	if !roles.HasRole(getServiceRoleName(authType, accessType, handle)) {
		return getServiceRoleDescription(authType, accessType, handle)
	}

	return ""
}

func CheckUserRoles(
	roles *tvm.ConsumerRoles,
	accessType AccessTypeSlug,
	handle HandlerSlug,
) string {
	if roles.HasRole(getUserRoleName(accessType, allHandlers)) {
		return ""
	}

	if !roles.HasRole(getUserRoleName(accessType, handle)) {
		return getUserRoleDescription(accessType, handle)
	}

	return ""
}

func getServiceRoleName(authType AuthTypeSlug, accessType AccessTypeSlug, handle HandlerSlug) string {
	return fmt.Sprintf(
		"/role/service/auth_type/%s/access_type/%s/handlers/%s/",
		authType,
		accessType,
		handle,
	)
}

func getServiceRoleDescription(authType AuthTypeSlug, accessType AccessTypeSlug, handle HandlerSlug) string {
	return fmt.Sprintf(
		"service '%s' access to '%s' handler with auth type '%s'",
		accessType,
		handle,
		authType,
	)
}

func getUserRoleName(accessType AccessTypeSlug, handle HandlerSlug) string {
	return fmt.Sprintf(
		"/role/user/access_type/%s/handlers/%s/",
		accessType,
		handle,
	)
}

func getUserRoleDescription(accessType AccessTypeSlug, handle HandlerSlug) string {
	return fmt.Sprintf(
		"user '%s' access to '%s' handler",
		accessType,
		handle,
	)
}
