package yasmsinternal

import (
	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/yandex/blackbox"
	"a.yandex-team.ru/library/go/yandex/blackbox/httpbb"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/errs"
	"a.yandex-team.ru/passport/shared/golibs/logger"
	"a.yandex-team.ru/passport/shared/golibs/utils"
)

type BlackBoxConfig struct {
	BlackboxURL string         `json:"url"`
	Timeout     utils.Duration `json:"timeout"`
	Retries     int            `json:"retries"`
}

func CreateBlackBox(cfg *BlackBoxConfig, tvmClient tvm.Client, tvmid tvm.ClientID) (*httpbb.Client, error) {
	bbClient, err := httpbb.NewClient(
		httpbb.Environment{
			BlackboxHost: cfg.BlackboxURL,
			TvmID:        tvmid,
		},
		httpbb.WithTVM(tvmClient),
		httpbb.WithTimeout(cfg.Timeout.Duration),
		httpbb.WithRetries(cfg.Retries),
	)
	if err != nil {
		return nil, xerrors.Errorf("Failed to create blackbox client: %w", err)
	}

	return bbClient, nil
}

func getUserByTvmUserTicket(ctx echo.Context, bb *httpbb.Client) (*blackbox.User, error) {
	ticket := ctx.Request().Header.Get("X-Ya-User-Ticket")
	if ticket == "" {
		return nil, &errs.UnauthorizedError{
			Status:    errs.Error,
			Message:   "User ticket is missing",
			Component: errs.CommonComponent,
		}
	}

	response, err := bb.UserTicket(
		ctx.Request().Context(),
		blackbox.UserTicketRequest{
			UserTicket: ticket,
		},
	)

	if err != nil {
		logger.Log().Errorf("BB returned error: %s", err)
		return nil, &errs.UnauthorizedError{
			Status:    errs.Error,
			Message:   "bb returned error",
			Component: errs.CommonComponent,
		}
	}

	if len(response.Users) != 1 {
		logger.Log().Errorf("BB problem with users num: %d", len(response.Users))
		return nil, &errs.UnauthorizedError{
			Status:    errs.Error,
			Message:   "bb returned bad result",
			Component: errs.CommonComponent,
		}
	}

	return &response.Users[0], nil
}
