package yasmsinternal

import (
	"encoding/json"
	"fmt"
	"io"
	"net/http"
	"strconv"

	"github.com/labstack/echo/v4"
	"github.com/santhosh-tekuri/jsonschema/v5"

	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/yasmsinternal/processor"
	"a.yandex-team.ru/passport/shared/golibs/juggler"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

const xYaReadOnly = "X-Ya-Read-Only"

func (t *YasmsInternal) HandleHealthCheck() echo.HandlerFunc {
	return func(ctx echo.Context) error {
		status := juggler.NewStatusOk()
		status.Update(t.processor.GetJugglerStatus())
		return ctx.String(http.StatusOK, status.String())
	}
}

func (t *YasmsInternal) HandleGetRegionsV1() echo.HandlerFunc {
	return func(ctx echo.Context) error {
		params := processor.GetRegionsRequest{}
		err := ctx.Bind(&params)
		if err != nil {
			return &errs.BadRequestError{
				Status:  errs.Error,
				Message: fmt.Sprintf("invalid params: %s", err.Error()),

				Component: errs.CommonComponent,
			}
		}

		regions, err := t.processor.HandleGetRegions(ctx.Request().Context(), params)
		if err != nil {
			return err
		}
		return ctx.JSON(http.StatusOK, regions)
	}
}

func (t *YasmsInternal) HandleRouteEnumsV1() echo.HandlerFunc {
	return func(ctx echo.Context) error {
		enums, err := t.processor.HandleRouteEnums(ctx.Request().Context())
		if err != nil {
			return err
		}
		return ctx.JSON(http.StatusOK, enums)
	}
}

func (t *YasmsInternal) HandleGetRoutesV1() echo.HandlerFunc {
	return func(ctx echo.Context) error {
		params := processor.GetRoutesRequest{}
		err := ctx.Bind(&params)
		if err != nil {
			return &errs.BadRequestError{
				Status:  errs.Error,
				Message: fmt.Sprintf("invalid params: %s", err.Error()),

				Component: errs.CommonComponent,
			}
		}

		params.Path = ctx.Request().URL.Path

		routes, err := t.processor.HandleGetRoutes(ctx.Request().Context(), params)
		if err != nil {
			return err
		}
		return ctx.JSON(http.StatusOK, routes)
	}
}

func (t *YasmsInternal) HandleGetGatesV1() echo.HandlerFunc {
	return func(ctx echo.Context) error {
		params := processor.GetGatesRequest{}
		err := ctx.Bind(&params)
		if err != nil {
			return &errs.BadRequestError{
				Status:  errs.Error,
				Message: fmt.Sprintf("invalid params: %s", err.Error()),

				Component: errs.CommonComponent,
			}
		}

		params.Path = ctx.Request().URL.Path

		gates, err := t.processor.HandleGetGates(ctx.Request().Context(), params)
		if err != nil {
			return err
		}
		return ctx.JSON(http.StatusOK, gates)
	}
}

func (t *YasmsInternal) HandleGetBlockedPhonesV1() echo.HandlerFunc {
	return func(ctx echo.Context) error {
		params := processor.GetBlockedPhonesRequest{}
		err := ctx.Bind(&params)
		if err != nil {
			return &errs.BadRequestError{
				Status:  errs.Error,
				Message: fmt.Sprintf("invalid params: %s", err.Error()),

				Component: errs.CommonComponent,
			}
		}

		params.Path = ctx.Request().URL.Path

		blockedPhones, err := t.processor.HandleGetBlockedPhones(ctx.Request().Context(), params)
		if err != nil {
			return err
		}
		return ctx.JSON(http.StatusOK, blockedPhones)
	}
}

func (t *YasmsInternal) HandleGetFallbacksV1() echo.HandlerFunc {
	return func(ctx echo.Context) error {
		params := processor.GetFallbacksRequest{}
		err := ctx.Bind(&params)
		if err != nil {
			return &errs.BadRequestError{
				Status:  errs.Error,
				Message: fmt.Sprintf("invalid params: %s", err.Error()),

				Component: errs.CommonComponent,
			}
		}

		params.Path = ctx.Request().URL.Path

		fallbacks, err := t.processor.HandleGetFallbacks(ctx.Request().Context(), params)
		if err != nil {
			return err
		}
		return ctx.JSON(http.StatusOK, fallbacks)
	}
}

func (t *YasmsInternal) HandleGetTemplatesV1() echo.HandlerFunc {
	return func(ctx echo.Context) error {
		params := processor.GetTemplatesRequest{}
		err := ctx.Bind(&params)
		if err != nil {
			return &errs.BadRequestError{
				Status:  errs.Error,
				Message: fmt.Sprintf("invalid params: %s", err.Error()),

				Component: errs.CommonComponent,
			}
		}

		params.Path = ctx.Request().URL.Path

		templates, err := t.processor.HandleGetTemplates(ctx.Request().Context(), params)
		if err != nil {
			return err
		}
		return ctx.JSON(http.StatusOK, templates)
	}
}

func (t *YasmsInternal) HandleSetRegionsV1() echo.HandlerFunc {
	return func(ctx echo.Context) error {
		body, err := io.ReadAll(ctx.Request().Body)
		if err != nil {
			return &errs.UnknownError{
				Status:  errs.Error,
				Message: "failed to read request body",
			}
		}
		err = validate(&setRegionsSchema, body)
		if err != nil {
			return err
		}

		var request processor.SetRegionsRequest

		err = json.Unmarshal(body, &request)
		if err != nil {
			logger.Log().Warnf("invalid json %s, %s", err.Error(), body)
			return &errs.BadRequestError{
				Status:  errs.Error,
				Message: "invalid json structure",
			}
		}

		request.MetaData.Author, err = t.GetChangeInfoAuthor(ctx)
		if err != nil {
			return err
		}

		response, err := t.processor.HandleSetRegions(ctx.Request().Context(), request, t.ModificationLog(ctx))
		if err != nil {
			return &errs.BadRequestError{
				Status:  errs.Error,
				Message: "failed to update database",
			}
		}
		return ctx.JSON(http.StatusOK, response)
	}
}

func (t *YasmsInternal) GetChangeInfoAuthor(ctx echo.Context) (string, error) {
	user, err := getUserByTvmUserTicket(ctx, t.bbClient)
	if err != nil {
		ctxlog.Warnf(ctx.Request().Context(), logger.Log(), "BB error: %s", err.Error())
		return "", &errs.BadRequestError{
			Status:  errs.Error,
			Message: "failed to get user from BB",
		}
	}
	ctxlog.Debugf(ctx.Request().Context(), logger.Log(), "BB returned login %s", user.Login)

	return user.Login, nil
}

func (t *YasmsInternal) HandleSetRoutesV1() echo.HandlerFunc {
	return func(ctx echo.Context) error {
		var request processor.SetRoutesRequest
		body, err := io.ReadAll(ctx.Request().Body)
		if err != nil {
			return &errs.UnknownError{
				Status:  errs.Error,
				Message: "failed to read request body",
			}
		}

		err = validate(&setRoutesSchema, body)
		if err != nil {
			return err
		}

		err = json.Unmarshal(body, &request)
		if err != nil {
			logger.Log().Warnf(err.Error())
			return &errs.BadRequestError{
				Status:  errs.Error,
				Message: "invalid json structure",
			}
		}

		request.MetaData.Author, err = t.GetChangeInfoAuthor(ctx)
		if err != nil {
			return err
		}
		response, err := t.processor.HandleSetRoutes(ctx.Request().Context(), request, t.ModificationLog(ctx))
		if err != nil {
			return &errs.BadRequestError{
				Status:  errs.Error,
				Message: "failed to update database",
			}
		}
		return ctx.JSON(http.StatusOK, response)
	}
}

func (t *YasmsInternal) HandleSetGatesV1() echo.HandlerFunc {
	return func(ctx echo.Context) error {
		var request processor.SetGatesRequest
		body, err := io.ReadAll(ctx.Request().Body)
		if err != nil {
			return &errs.UnknownError{
				Status:  errs.Error,
				Message: "failed to read request body",
			}
		}

		err = validate(&setGatesSchema, body)
		if err != nil {
			return err
		}

		err = json.Unmarshal(body, &request)
		if err != nil {
			logger.Log().Warnf(err.Error())
			return &errs.BadRequestError{
				Status:  errs.Error,
				Message: "invalid json structure",
			}
		}

		request.MetaData.Author, err = t.GetChangeInfoAuthor(ctx)
		if err != nil {
			return err
		}

		response, err := t.processor.HandleSetGates(ctx.Request().Context(), request, t.ModificationLog(ctx))
		if err != nil {
			return &errs.BadRequestError{
				Status:  errs.Error,
				Message: "failed to update database",
			}
		}
		return ctx.JSON(http.StatusOK, response)
	}
}

func (t *YasmsInternal) HandleSetBlockedPhonesV1() echo.HandlerFunc {
	return func(ctx echo.Context) error {
		var request processor.SetBlockedPhonesRequest
		body, err := io.ReadAll(ctx.Request().Body)
		if err != nil {
			return &errs.UnknownError{
				Status:  errs.Error,
				Message: "failed to read request body",
			}
		}

		err = validate(&setBlockedPhonesSchema, body)
		if err != nil {
			return err
		}

		err = json.Unmarshal(body, &request)
		if err != nil {
			logger.Log().Warnf(err.Error())
			return &errs.BadRequestError{
				Status:  errs.Error,
				Message: "invalid json structure",
			}
		}

		request.MetaData.Author, err = t.GetChangeInfoAuthor(ctx)
		if err != nil {
			return err
		}

		response, err := t.processor.HandleSetBlockedPhones(ctx.Request().Context(), request, t.ModificationLog(ctx))
		if err != nil {
			return &errs.BadRequestError{
				Status:  errs.Error,
				Message: "failed to update database",
			}
		}
		return ctx.JSON(http.StatusOK, response)
	}
}

func (t *YasmsInternal) HandleSetFallbacksV1() echo.HandlerFunc {
	return func(ctx echo.Context) error {
		var request processor.SetFallbacksRequest
		body, err := io.ReadAll(ctx.Request().Body)
		if err != nil {
			return &errs.UnknownError{
				Status:  errs.Error,
				Message: "failed to read request body",
			}
		}

		err = validate(&setFallbacksSchema, body)
		if err != nil {
			return err
		}

		err = json.Unmarshal(body, &request)
		if err != nil {
			logger.Log().Errorf(err.Error())
			return &errs.BadRequestError{
				Status:  errs.Error,
				Message: "invalid json structure",
			}
		}

		request.MetaData.Author, err = t.GetChangeInfoAuthor(ctx)
		if err != nil {
			return err
		}

		response, err := t.processor.HandleSetFallbacks(ctx.Request().Context(), request, t.ModificationLog(ctx))
		if err != nil {
			return &errs.BadRequestError{
				Status:  errs.Error,
				Message: "failed to update database",
			}
		}
		return ctx.JSON(http.StatusOK, response)
	}
}

func (t *YasmsInternal) HandleSetTemplatesV1() echo.HandlerFunc {
	return func(ctx echo.Context) error {
		var request processor.SetTemplatesRequest
		body, err := io.ReadAll(ctx.Request().Body)
		if err != nil {
			return &errs.UnknownError{
				Status:  errs.Error,
				Message: "failed to read request body",
			}
		}

		err = validate(&setTemplatesSchema, body)
		if err != nil {
			return err
		}

		err = json.Unmarshal(body, &request)
		if err != nil {
			logger.Log().Errorf(err.Error())
			return &errs.BadRequestError{
				Status:  errs.Error,
				Message: "invalid json structure",
			}
		}

		request.MetaData.Author, err = t.GetChangeInfoAuthor(ctx)
		if err != nil {
			return err
		}

		response, err := t.processor.HandleSetTemplates(ctx.Request().Context(), request, t.ModificationLog(ctx))
		if err != nil {
			return &errs.BadRequestError{
				Status:  errs.Error,
				Message: "failed to update database",
			}
		}
		return ctx.JSON(http.StatusOK, response)
	}
}

func (t *YasmsInternal) HandleTemplatesParseV1() echo.HandlerFunc {
	return func(ctx echo.Context) error {
		body, err := io.ReadAll(ctx.Request().Body)
		if err != nil {
			return &errs.UnknownError{
				Status:  errs.Error,
				Message: "failed to read request body",
			}
		}

		response, err := t.processor.ValidateTemplateSyntax(string(body))
		if err != nil {
			return err
		}

		return ctx.JSON(http.StatusOK, response)
	}
}

func validate(schema *jsonschema.Schema, body []byte) error {
	makeError := func(err error) error {
		return &errs.BadRequestError{
			Status:  errs.Error,
			Message: fmt.Sprintf("invalid json: %s", err.Error()),
		}
	}

	var v interface{}
	err := json.Unmarshal(body, &v)
	if err != nil {
		return makeError(err)
	}
	err = schema.Validate(v)
	if err != nil {
		return makeError(err)
	}

	return nil
}

func (t *YasmsInternal) HandleGetAuditBulkInfoV1() echo.HandlerFunc {
	return func(ctx echo.Context) error {
		params := processor.GetAuditBulkInfoRequest{}
		err := ctx.Bind(&params)
		if err != nil {
			return &errs.BadRequestError{
				Status:    errs.Error,
				Message:   fmt.Sprintf("invalid params: %s", err.Error()),
				Component: errs.CommonComponent,
			}
		}

		if _, err = strconv.ParseUint(params.BulkID, 10, 64); err != nil {
			return &errs.BadRequestError{
				Status:    errs.Error,
				Message:   fmt.Sprintf("invalid params: %s", err.Error()),
				Component: errs.CommonComponent,
			}
		}

		bulkInfo, err := t.processor.HandleAuditBulkInfo(ctx.Request().Context(), params)
		if err != nil {
			return err
		}
		return ctx.JSON(http.StatusOK, bulkInfo)
	}
}

func (t *YasmsInternal) HandleGetAuditChangeInfoV1() echo.HandlerFunc {
	return func(ctx echo.Context) error {
		genInvalidParamsErr := func(errStr string) error {
			return &errs.BadRequestError{
				Status:    errs.Error,
				Message:   fmt.Sprintf("invalid params: %s", errStr),
				Component: errs.CommonComponent,
			}
		}
		params := processor.GetAuditChangeInfoRequest{}
		err := ctx.Bind(&params)
		if err != nil {
			return genInvalidParamsErr(err.Error())
		}

		if len(params.ChangeIDs) == 0 {
			return genInvalidParamsErr("no change_id params")
		}

		for _, id := range params.ChangeIDs {
			if _, err = strconv.ParseUint(id, 10, 64); err != nil {
				return genInvalidParamsErr(fmt.Sprintf("%s should be int", id))
			}
		}

		bulkInfo, err := t.processor.HandleAuditChangeInfo(ctx.Request().Context(), params)
		if err != nil {
			return err
		}
		return ctx.JSON(http.StatusOK, bulkInfo)
	}
}
