package yasmsinternal

import (
	"fmt"
	"net/http"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/errs"
	"a.yandex-team.ru/passport/shared/golibs/httpdaemon/middlewares"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

const (
	headerRequestID = "X-Request-Id"
)

func (t *YasmsInternal) handleErrorMiddleware() echo.MiddlewareFunc {
	return func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(ctx echo.Context) error {
			err := next(ctx)
			switch e := err.(type) {
			case nil:
				return nil
			case *errs.UnauthorizedError:
				logger.Log().Warn(e.Error())
				return ctx.JSON(http.StatusUnauthorized, e)
			case *errs.AccessDeniedError:
				logger.Log().Warn(e.Error())
				return ctx.JSON(http.StatusForbidden, e)
			case *errs.BadRequestError:
				logger.Log().Warn(e.Error())
				return ctx.JSON(http.StatusBadRequest, e)
			case *errs.UnknownError:
				logger.Log().Error(e.Error())
				return ctx.JSON(http.StatusInternalServerError, e)
			case *echo.HTTPError:
				logger.Log().Error(e.Error())
				return ctx.JSON(e.Code, e)
			}

			logger.Log().Error(err.Error())
			return ctx.JSON(
				http.StatusInternalServerError,
				&errs.UnknownError{
					Status:  errs.Error,
					Message: "Internal Error",

					Component: errs.CommonComponent,
				},
			)
		}
	}
}

func (t *YasmsInternal) accessLogMiddleware() echo.MiddlewareFunc {
	return func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(c echo.Context) (err error) {
			c.Response().After(func() {
				startTime := middlewares.ContextStartInstant(c.Request().Context())

				st := tvm.ContextServiceTicket(c.Request().Context())
				ut := tvm.ContextUserTicket(c.Request().Context())

				srcID := "-"
				if st != nil {
					srcID = fmt.Sprintf("%d", st.SrcID)
				}

				defaultUID := "-"
				if ut != nil {
					defaultUID = fmt.Sprintf("%d", ut.DefaultUID)
				}

				query := c.Request().URL.RawQuery
				if query == "" {
					query = "-"
				}

				t.accessLog.Debugf(
					"%s\t%s\t%d\t%.2fms\ttvmid=%s\tuser=%s\t%s\t%s\t%s",
					middlewares.ContextReqID(c.Request().Context()),
					c.RealIP(),
					c.Response().Status,
					float64(time.Since(startTime).Microseconds())/1000,
					srcID,
					defaultUID,
					c.Request().Method,
					c.Request().URL.Path,
					query,
				)
			})

			return next(c)
		}
	}
}

func (t *YasmsInternal) sendReqIDMiddleware() echo.MiddlewareFunc {
	return func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(c echo.Context) (err error) {
			c.Response().Header().Set(
				headerRequestID,
				middlewares.ContextReqID(c.Request().Context()),
			)

			return next(c)
		}
	}
}
