package processor

import (
	"context"
	"time"

	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/filter"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/loggers"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/model"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

type (
	GetBlockedPhonesRequest struct {
		FromID model.EntityID `query:"min"`
		Limit  uint64         `query:"limit"`
		Filter string         `query:"filter"`

		Path string
	}

	GetBlockedPhonesResponse struct {
		BlockedPhones []*model.BlockedPhone `json:"blockedphones"`
		TotalCount    uint64                `json:"total_count"`
		NextURL       string                `json:"next"`
	}
)

type CreateBlockedPhonesEntry = model.BlockedPhone

type UpdateBlockedPhonesEntry = model.BlockedPhone

type (
	SetBlockedPhonesRequest struct {
		Delete   []model.EntityID            `json:"delete"`
		Create   []*CreateBlockedPhonesEntry `json:"create"`
		Update   []*UpdateBlockedPhonesEntry `json:"update"`
		MetaData model.AuditLogBulkParams    `json:"change_info"`
	}

	SetBlockedPhonesResponse struct {
		Status  errs.RequestStatus `json:"status"`
		Message string             `json:"message"`
	}
)

var blockedPhonesFilterSchema = &filter.Schema{
	Fields: filter.FieldsSchema{
		model.BlockidAlias:   filter.NumberFieldSchema,
		model.BlocktillAlias: filter.DatetimeFieldSchema,
		model.BlocktypeAlias: filter.EnumFieldSchema(map[string]interface{}{
			model.BlockTypePermanent: nil,
			model.BlockTypeTemporary: nil,
		}),
		model.PhoneAlias: filter.StringFieldSchema,
	},
	CheckDepth: true,
	MaxDepth:   1,
}

func (processor *Processor) HandleGetBlockedPhones(
	ctx context.Context,
	request GetBlockedPhonesRequest,
) (*GetBlockedPhonesResponse, error) {
	if request.Limit <= 0 {
		return nil, &errs.BadRequestError{
			Status:  errs.Error,
			Message: "limit must be a natural number",

			Component: errs.ProcessorComponent,
		}
	}

	blockedPhonesFilter, err := GetSelectFilter(request.Filter, blockedPhonesFilterSchema)
	if err != nil {
		return nil, err
	}

	response := &GetBlockedPhonesResponse{}

	response.BlockedPhones, err = processor.fetcher.GetBlockedPhones(ctx, request.FromID, request.Limit, blockedPhonesFilter)
	if err != nil {
		logger.Log().Warnf(
			"failed fetch blockedphones from %s, with limit %d: %s", request.FromID, request.Limit, err.Error())
		return nil, &errs.UnknownError{
			Status:  errs.Error,
			Message: "failed to fetch blockedphones",

			Component: errs.ProcessorComponent,
		}
	}

	response.TotalCount, err = processor.fetcher.GetBlockedPhonesCount(ctx, blockedPhonesFilter)
	if err != nil {
		logger.Log().Warnf("failed fetch blockedphones count: %s", err.Error())
		return nil, &errs.UnknownError{
			Status:  errs.Error,
			Message: "failed to fetch blockedphones count",

			Component: errs.ProcessorComponent,
		}
	}
	response.TotalCount = max(response.TotalCount, uint64(len(response.BlockedPhones)))

	if len(response.BlockedPhones) == int(request.Limit) {
		response.NextURL = makeNextURLWithFilter(request.Path, response.BlockedPhones[len(response.BlockedPhones)-1].ID, request.Limit, request.Filter)
	}

	return response, nil
}

func (processor *Processor) HandleSetBlockedPhones(
	ctx context.Context,
	request SetBlockedPhonesRequest,
	modificationLog *loggers.TskvContextLog,
) (*SetBlockedPhonesResponse, error) {
	err := processor.writer.SetBlockedPhones(ctx, request.Delete, request.Create, request.Update, &request.MetaData)
	if err != nil {
		logger.Log().Warnf("mysql error: %s", err.Error())
		return nil, &errs.UnknownError{
			Status:  errs.Error,
			Message: "failed to modify entries",
		}
	}

	modificationLog.LogBlockedPhonesAction(time.Now(), processor.writer.GetName(), request.Delete, request.Create, request.Update)

	return &SetBlockedPhonesResponse{
		Status:  errs.Ok,
		Message: "",
	}, nil
}
