package processor

import (
	"context"
	"time"

	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/filter"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/loggers"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/model"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

type (
	GetFallbacksRequest struct {
		FromID model.EntityID `query:"min"`
		Limit  uint64         `query:"limit"`
		Filter string         `query:"filter"`

		Path string
	}

	GetFallbacksResponse struct {
		Fallbacks  []*model.Fallback `json:"fallbacks"`
		TotalCount uint64            `json:"total_count"`
		NextURL    string            `json:"next"`
	}
)

type CreateFallbacksEntry = model.Fallback

type UpdateFallbacksEntry = model.Fallback

type (
	SetFallbacksRequest struct {
		Delete   []model.EntityID         `json:"delete"`
		Create   []*CreateFallbacksEntry  `json:"create"`
		Update   []*UpdateFallbacksEntry  `json:"update"`
		MetaData model.AuditLogBulkParams `json:"change_info"`
	}

	SetFallbacksResponse struct {
		Status  errs.RequestStatus `json:"status"`
		Message string             `json:"message"`
	}
)

var fallbacksFilterSchema = &filter.Schema{
	Fields: filter.FieldsSchema{
		model.FallBackIDAlias: filter.NumberFieldSchema,
		model.SrcgateAlias:    filter.StringFieldSchema,
		model.SrcnameAlias:    filter.StringFieldSchema,
		model.DstgateAlias:    filter.StringFieldSchema,
		model.DstnameAlias:    filter.StringFieldSchema,
		model.OrderAlias:      filter.NumberFieldWithCompareSchema,
	},
	CheckDepth: true,
	MaxDepth:   1,
}

func (processor *Processor) HandleGetFallbacks(
	ctx context.Context,
	request GetFallbacksRequest,
) (*GetFallbacksResponse, error) {
	if request.Limit <= 0 {
		return nil, &errs.BadRequestError{
			Status:  errs.Error,
			Message: "limit must be a natural number",

			Component: errs.ProcessorComponent,
		}
	}

	fallbacksFilter, err := GetSelectFilter(request.Filter, fallbacksFilterSchema)
	if err != nil {
		return nil, err
	}

	response := &GetFallbacksResponse{}

	response.Fallbacks, err = processor.fetcher.GetFallbacks(ctx, request.FromID, request.Limit, fallbacksFilter)
	if err != nil {
		logger.Log().Errorf(
			"failed fetch fallbacks from %s, with limit %d: %s", request.FromID, request.Limit, err.Error())
		return nil, &errs.UnknownError{
			Status:  errs.Error,
			Message: "failed to fetch fallbacks",

			Component: errs.ProcessorComponent,
		}
	}

	response.TotalCount, err = processor.fetcher.GetFallbacksCount(ctx, fallbacksFilter)
	if err != nil {
		logger.Log().Warnf("failed fetch fallbacks count: %s", err.Error())
		return nil, &errs.UnknownError{
			Status:  errs.Error,
			Message: "failed to fetch fallbacks count",

			Component: errs.ProcessorComponent,
		}
	}
	response.TotalCount = max(response.TotalCount, uint64(len(response.Fallbacks)))

	if len(response.Fallbacks) == int(request.Limit) {
		response.NextURL = makeNextURLWithFilter(request.Path, response.Fallbacks[len(response.Fallbacks)-1].ID, request.Limit, request.Filter)
	}

	return response, nil
}

func (processor *Processor) HandleSetFallbacks(
	ctx context.Context,
	request SetFallbacksRequest,
	modificationLog *loggers.TskvContextLog,
) (*SetFallbacksResponse, error) {
	err := processor.writer.SetFallbacks(ctx, request.Delete, request.Create, request.Update, &request.MetaData)
	if err != nil {
		logger.Log().Errorf("mysql error: %s", err.Error())
		return nil, &errs.UnknownError{
			Status:  errs.Error,
			Message: "failed to modify entries",
		}
	}

	modificationLog.LogFallbacksAction(time.Now(), processor.writer.GetName(), request.Delete, request.Create, request.Update)

	return &SetFallbacksResponse{
		Status:  errs.Ok,
		Message: "",
	}, nil
}
