package processor

import (
	"context"
	"time"

	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/filter"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/loggers"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/model"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

type (
	GetGatesRequest struct {
		FromID model.EntityID `query:"min"`
		Limit  uint64         `query:"limit"`
		Filter string         `query:"filter"`

		Path string
	}

	GetGatesResponse struct {
		Gates      []*model.GateWithAudit `json:"gates"`
		TotalCount uint64                 `json:"total_count"`
		NextURL    string                 `json:"next"`
	}
)

type CreateGateEntry = model.Gate

type UpdateGateEntry = model.Gate

type (
	SetGatesRequest struct {
		Delete   []model.EntityID         `json:"delete"`
		Create   []*CreateGateEntry       `json:"create"`
		Update   []*UpdateGateEntry       `json:"update"`
		MetaData model.AuditLogBulkParams `json:"change_info"`
	}

	SetGatesResponse struct {
		Status  errs.RequestStatus `json:"status"`
		Message string             `json:"message"`
	}
)

var gatesFilterSchema = &filter.Schema{
	Fields: filter.FieldsSchema{
		model.GateIDFieldAlias:   filter.NumberFieldSchema,
		model.AliaseFieldAlias:   filter.StringFieldSchema,
		model.FromnameFieldAlias: filter.StringFieldSchema,
	},
	CheckDepth: true,
	MaxDepth:   1,
}

func (processor *Processor) HandleGetGates(ctx context.Context, request GetGatesRequest) (*GetGatesResponse, error) {
	if request.Limit <= 0 {
		return nil, &errs.BadRequestError{
			Status:  errs.Error,
			Message: "limit must be a natural number",

			Component: errs.ProcessorComponent,
		}
	}
	gatesFilter, err := GetSelectFilter(request.Filter, gatesFilterSchema)
	if err != nil {
		return nil, err
	}

	response := &GetGatesResponse{}

	response.Gates, err = processor.fetcher.GetGates(ctx, request.FromID, request.Limit, gatesFilter)
	if err != nil {
		logger.Log().Warnf(
			"failed fetch gates from %s, with limit %d: %s", request.FromID, request.Limit, err.Error())
		return nil, &errs.UnknownError{
			Status:  errs.Error,
			Message: "failed to fetch gates",

			Component: errs.ProcessorComponent,
		}
	}

	response.TotalCount, err = processor.fetcher.GetGatesCount(ctx, gatesFilter)
	if err != nil {
		logger.Log().Warnf("failed fetch gates count: %s", err.Error())
		return nil, &errs.UnknownError{
			Status:  errs.Error,
			Message: "failed to fetch gates count",

			Component: errs.ProcessorComponent,
		}
	}
	response.TotalCount = max(response.TotalCount, uint64(len(response.Gates)))

	if len(response.Gates) == int(request.Limit) {
		response.NextURL = makeNextURLWithFilter(request.Path, response.Gates[len(response.Gates)-1].ID, request.Limit, request.Filter)
	}

	return response, nil
}

func (processor *Processor) HandleSetGates(
	ctx context.Context,
	request SetGatesRequest,
	modificationLog *loggers.TskvContextLog,
) (*SetGatesResponse, error) {
	err := processor.writer.SetGates(ctx, request.Delete, request.Create, request.Update, &request.MetaData)
	if err != nil {
		logger.Log().Warnf("mysql error: %s", err.Error())
		return nil, &errs.UnknownError{
			Status:  errs.Error,
			Message: "failed to modify entries",
		}
	}

	modificationLog.LogGatesAction(time.Now(), processor.writer.GetName(), request.Delete, request.Create, request.Update)
	processor.ytUpdater.RunNow()

	return &SetGatesResponse{
		Status:  errs.Ok,
		Message: "",
	}, nil
}
