package processor

import (
	"context"
	"fmt"
	"time"

	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/filter"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/loggers"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/model"
	"a.yandex-team.ru/passport/infra/libs/go/task"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

type (
	GetRegionsRequest struct {
		Filter string `query:"filter"`
	}

	RegionsResponse struct {
		Regions []*model.Region `json:"regions"`
	}
)

type (
	SetRegionsRequest struct {
		Delete   []model.EntityID         `json:"delete"`
		Create   []*model.Region          `json:"create"`
		Update   []*model.Region          `json:"update"`
		MetaData model.AuditLogBulkParams `json:"change_info"`
	}

	SetRegionsResponse struct {
		Status  errs.RequestStatus `json:"status"`
		Message string             `json:"message"`
	}
)

var regionsFilterSchema = &filter.Schema{
	Fields: filter.FieldsSchema{
		model.PrefixFieldAlias: filter.StringFieldSchema,
		model.NameFieldAlias:   filter.StringFieldSchema,
	},
	CheckDepth: true,
	MaxDepth:   1,
}

func (processor *Processor) HandleGetRegions(ctx context.Context, request GetRegionsRequest) (*RegionsResponse, error) {
	regionsFilter, err := GetSelectFilter(request.Filter, regionsFilterSchema)
	if err != nil {
		return nil, err
	}

	response := &RegionsResponse{}

	response.Regions, err = processor.fetcher.GetRegions(ctx, regionsFilter)
	if err != nil {
		logger.Log().Warnf("failed fetch regions: %s", err.Error())
		return nil, &errs.UnknownError{
			Status:  errs.Error,
			Message: "failed to fetch regions",

			Component: errs.ProcessorComponent,
		}
	}

	return response, nil
}

func (processor *Processor) HandleSetRegions(
	ctx context.Context,
	request SetRegionsRequest,
	modificationLog *loggers.TskvContextLog,
) (*SetRegionsResponse, error) {
	err := processor.writer.SetRegions(ctx, request.Delete, request.Create, request.Update, &request.MetaData)
	if err != nil {
		logger.Log().Warnf("failed to modify regions: %s", err.Error())
		return nil, &errs.UnknownError{
			Status:  errs.Error,
			Message: "failed to modify regions",
		}
	}

	modificationLog.LogRegionsAction(time.Now(), processor.writer.GetName(), request.Delete, request.Create, request.Update)
	processor.ytUpdater.RunNow()

	return &SetRegionsResponse{
		Status:  errs.Ok,
		Message: "",
	}, nil
}

func (processor *Processor) updateYtTablesTask() (task.Status, error) {
	logger.Log().Debug("Yt update started")

	regionsDB, err := processor.fetcher.GetRegions(context.Background(), nil)
	if err != nil {
		return task.StatusFailed, fmt.Errorf("failed to select regions: %s", err)
	}
	err = processor.ytTablesProvider.SaveRegions(context.Background(), regionsDB)
	if err != nil {
		return task.StatusFailed, fmt.Errorf("failed to save regions to yt: %s", err)
	}

	gatesDB, err := processor.fetcher.GetAllGates(context.Background())
	if err != nil {
		return task.StatusFailed, fmt.Errorf("failed to select gates: %s", err)
	}
	err = processor.ytTablesProvider.SaveGates(context.Background(), gatesDB)
	if err != nil {
		return task.StatusFailed, fmt.Errorf("failed to save gates to yt: %s", err)
	}

	logger.Log().Debug("Yt update completed")

	return task.StatusSuccess, nil
}
