package processor

import (
	"context"
	"fmt"
	"time"

	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/filter"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/loggers"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/model"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

type (
	GetRoutesRequest struct {
		FromID model.EntityID `query:"min"`
		Limit  uint64         `query:"limit"`
		Filter string         `query:"filter"`

		Path string
	}

	GetRoutesResponse struct {
		Routes     []*model.RouteFullInfo `json:"routes"`
		TotalCount uint64                 `json:"total_count"`
		NextURL    string                 `json:"next"`
	}
)

type (
	SetRoutesRequest struct {
		Delete   []model.EntityID         `json:"delete"`
		Create   []*model.Route           `json:"create"`
		Update   []*model.Route           `json:"update"`
		MetaData model.AuditLogBulkParams `json:"change_info"`
	}

	SetRoutesResponse struct {
		Status  errs.RequestStatus `json:"status"`
		Message string             `json:"message"`
	}
)

var routesFilterSchema = &filter.Schema{
	Fields: filter.FieldsSchema{
		model.GateIDFieldAlias:      filter.NumberFieldSchema,
		model.GateID2FieldAlias:     filter.NumberFieldSchema,
		model.GateID3FieldAlias:     filter.NumberFieldSchema,
		model.DestinationFieldAlias: filter.StringFieldSchema,
		model.ModeFieldAlias:        filter.StringFieldSchema,
		model.WeightFieldAlias:      filter.NumberFieldWithCompareSchema,
		model.AliaseFieldAlias:      filter.StringFieldSchema,
		model.Aliase2FieldAlias:     filter.StringFieldSchema,
		model.Aliase3FieldAlias:     filter.StringFieldSchema,
		model.FromnameFieldAlias:    filter.StringFieldSchema,
		model.Fromname2FieldAlias:   filter.StringFieldSchema,
		model.Fromname3FieldAlias:   filter.StringFieldSchema,
	},
	CheckDepth: true,
	MaxDepth:   1,
}

func (processor *Processor) HandleGetRoutes(ctx context.Context, request GetRoutesRequest) (*GetRoutesResponse, error) {
	if request.Limit <= 0 {
		return nil, &errs.BadRequestError{
			Status:  errs.Error,
			Message: "limit must be a natural number",

			Component: errs.ProcessorComponent,
		}
	}

	routesFilter, err := GetSelectFilter(request.Filter, routesFilterSchema)
	if err != nil {
		return nil, err
	}

	routesInfo, err := processor.fetcher.GetRoutesInfo(ctx, request.FromID, request.Limit, routesFilter)
	if err != nil {
		logger.Log().Warnf(
			"failed fetch routes from %s, with limit %d: %s", request.FromID, request.Limit, err.Error())
		return nil, &errs.UnknownError{
			Status:  errs.Error,
			Message: "failed to fetch routes",

			Component: errs.ProcessorComponent,
		}
	}

	routesCount, err := processor.fetcher.GetRoutesCount(ctx, routesFilter)
	if err != nil {
		logger.Log().Warnf("failed fetch routes count: %s", err.Error())
		return nil, &errs.UnknownError{
			Status:  errs.Error,
			Message: "failed to fetch routes count",

			Component: errs.ProcessorComponent,
		}
	}
	routesCount = max(routesCount, uint64(len(routesInfo)))

	response := &GetRoutesResponse{
		Routes:     make([]*model.RouteFullInfo, 0, len(routesInfo)),
		TotalCount: routesCount,
	}

	// make one request to db and use its result in loop
	allRegionsDB, err := processor.fetcher.GetRegions(ctx, nil)
	if err != nil {
		return nil, fmt.Errorf("failed to select regions: %s", err)
	}

	for _, routeInfo := range routesInfo {
		region, err := GetRoutesRegion(routeInfo.PhonePrefix, allRegionsDB)
		if err != nil {
			logger.Log().Warnf(
				"failed fetch region for %s: %s", routeInfo.PhonePrefix, err.Error())
			return nil, &errs.UnknownError{
				Status:  errs.Error,
				Message: fmt.Sprintf("failed to fetch region for %s", routeInfo.PhonePrefix),

				Component: errs.ProcessorComponent,
			}
		}

		response.Routes = append(response.Routes, &model.RouteFullInfo{
			RouteInfo: routeInfo,
			Region:    region,
		})
	}

	if len(response.Routes) == int(request.Limit) {
		lastID := response.Routes[len(response.Routes)-1].ID
		response.NextURL = makeNextURLWithFilter(request.Path, lastID, request.Limit, request.Filter)
	}

	return response, nil
}

func (processor *Processor) HandleSetRoutes(
	ctx context.Context,
	request SetRoutesRequest,
	modificationLog *loggers.TskvContextLog,
) (*SetRoutesResponse, error) {
	err := processor.writer.SetRoutes(ctx, request.Delete, request.Create, request.Update, &request.MetaData)
	if err != nil {
		logger.Log().Warnf("mysql error: %s", err.Error())
		return nil, &errs.UnknownError{
			Status:  errs.Error,
			Message: "failed to modify entries",
		}
	}

	modificationLog.LogRoutesAction(time.Now(), processor.writer.GetName(), request.Delete, request.Create, request.Update)

	return &SetRoutesResponse{
		Status:  errs.Ok,
		Message: "",
	}, nil
}
