package processor

import (
	"context"
	"fmt"
	"time"

	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/loggers"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/model"
	"a.yandex-team.ru/passport/infra/libs/go/yasms"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

type (
	GetTemplatesRequest struct {
		FromID model.EntityID `query:"min"`
		Limit  uint64         `query:"limit"`
		Filter string         `query:"filter"`

		Path string
	}

	GetTemplatesResponse struct {
		Templates  []*model.Template `json:"templates"`
		TotalCount uint64            `json:"total_count"`
		NextURL    string            `json:"next"`
	}
)

type CreateTemplatesEntry = model.Template

type UpdateTemplatesEntry = model.Template

type (
	SetTemplatesRequest struct {
		Create   []*CreateTemplatesEntry  `json:"create"`
		Update   []*UpdateTemplatesEntry  `json:"update"`
		MetaData model.AuditLogBulkParams `json:"change_info"`
	}

	SetTemplatesResponse struct {
		Status  errs.RequestStatus `json:"status"`
		Message string             `json:"message"`
	}
)

type (
	TemplatesParseResponse struct {
		FieldsList []string `json:"fields_list"`
	}
)

func (processor *Processor) HandleGetTemplates(
	ctx context.Context,
	request GetTemplatesRequest,
) (*GetTemplatesResponse, error) {
	if request.Limit <= 0 {
		return nil, &errs.BadRequestError{
			Status:  errs.Error,
			Message: "limit must be a natural number",

			Component: errs.ProcessorComponent,
		}
	}

	response := &GetTemplatesResponse{}
	var err error
	response.Templates, err = processor.fetcher.GetTemplates(ctx, request.FromID, request.Limit, nil)
	if err != nil {
		logger.Log().Errorf(
			"failed fetch templates from %s, with limit %d: %s", request.FromID, request.Limit, err.Error())
		return nil, &errs.UnknownError{
			Status:  errs.Error,
			Message: "failed to fetch templates",

			Component: errs.ProcessorComponent,
		}
	}

	response.TotalCount, err = processor.fetcher.GetTemplatesCount(ctx, nil)
	if err != nil {
		logger.Log().Warnf("failed fetch templates count: %s", err.Error())
		return nil, &errs.UnknownError{
			Status:  errs.Error,
			Message: "failed to fetch templates count",

			Component: errs.ProcessorComponent,
		}
	}
	response.TotalCount = max(response.TotalCount, uint64(len(response.Templates)))

	if len(response.Templates) == int(request.Limit) {
		response.NextURL = makeNextURLWithFilter(request.Path, response.Templates[len(response.Templates)-1].ID, request.Limit, request.Filter)
	}

	return response, nil
}

func (processor *Processor) HandleSetTemplates(
	ctx context.Context,
	request SetTemplatesRequest,
	modificationLog *loggers.TskvContextLog,
) (*SetTemplatesResponse, error) {
	err := processor.writer.SetTemplates(ctx, request.Create, request.Update, &request.MetaData)
	if err != nil {
		logger.Log().Errorf("mysql error: %s", err.Error())
		return nil, &errs.UnknownError{
			Status:  errs.Error,
			Message: "failed to modify entries",
		}
	}

	modificationLog.LogTemplatesAction(time.Now(), processor.writer.GetName(), request.Create, request.Update)

	return &SetTemplatesResponse{
		Status:  errs.Ok,
		Message: "",
	}, nil
}

func (processor *Processor) ValidateTemplateSyntax(template string) (*TemplatesParseResponse, error) {
	fields, err := yasms.GetYaSmsTemplateTokens(template)
	if err != nil {
		return nil, &errs.BadRequestError{
			Status:    errs.Error,
			Message:   fmt.Sprintf("failed to validate template: %s", err.Error()),
			Component: errs.ProcessorComponent,
		}
	}
	return &TemplatesParseResponse{FieldsList: fields}, nil
}
