package processor

import (
	"fmt"

	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/model"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/model/mysql"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/model/ytc"
	"a.yandex-team.ru/passport/infra/libs/go/task"
	"a.yandex-team.ru/passport/shared/golibs/juggler"
	"a.yandex-team.ru/passport/shared/golibs/utils"
)

type Driver = string

const (
	MySQL Driver = "mysql"
	YT    Driver = "yt"
)

type ProviderConfig struct {
	Driver      Driver               `json:"driver"`
	MySQLConfig mysql.ProviderConfig `json:"mysql_config"`
	YtConfig    ytc.ProviderConfig   `json:"yt_config"`
}

type Config struct {
	ReaderConfig  ProviderConfig `json:"reader"`
	WriterConfig  ProviderConfig `json:"writer"`
	RegionsConfig ProviderConfig `json:"regions"`

	UpdateInterval utils.Duration `json:"update_interval"`
}

type Processor struct {
	fetcher model.Reader
	writer  model.Writer

	ytTablesProvider model.YtTablesProvider
	ytUpdater        *task.RegularTask
}

func NewFetcher(config ProviderConfig) (model.Reader, error) {
	switch config.Driver {
	case MySQL:
		return mysql.NewMySQLProvider(config.MySQLConfig)
	}

	return nil, &errs.RuntimeError{
		Message:   fmt.Sprintf("invalid fetcher driver: %s", config.Driver),
		Component: errs.ProcessorComponent,
	}
}

func NewWriter(config ProviderConfig) (model.Writer, error) {
	switch config.Driver {
	case MySQL:
		return mysql.NewMySQLProvider(config.MySQLConfig)
	}

	return nil, &errs.RuntimeError{
		Message:   fmt.Sprintf("invalid writer driver: %s", config.Driver),
		Component: errs.ProcessorComponent,
	}
}

func NewProcessor(config Config, tvmClient tvm.Client) (*Processor, error) {
	fetcher, err := NewFetcher(config.ReaderConfig)
	if err != nil {
		return nil, err
	}

	writer, err := NewWriter(config.WriterConfig)
	if err != nil {
		return nil, err
	}

	ytTablesProvider, err := ytc.NewYtProvider(config.RegionsConfig.YtConfig, tvmClient)
	if err != nil {
		return nil, err
	}

	processor := &Processor{
		fetcher:          fetcher,
		writer:           writer,
		ytTablesProvider: ytTablesProvider,
	}
	processor.ytUpdater = task.NewRegularTask(
		processor.updateYtTablesTask,
		task.RegularTaskSettings{
			Name:           "yt-updater",
			Period:         config.UpdateInterval.Duration,
			BufferCapacity: 500,
		}).RunNow()

	return processor, nil
}

func (processor *Processor) Stop() {
	processor.ytUpdater.Stop()
}

func (processor *Processor) GetJugglerStatus() *juggler.Status {
	status := juggler.NewStatusOk()
	status.Update(processor.fetcher.GetJugglerStatus())
	status.Update(processor.writer.GetJugglerStatus())

	return status
}
