package yasmsinternal

import (
	"regexp"

	"github.com/santhosh-tekuri/jsonschema/v5"
)

var phoneNumberRegexp = regexp.MustCompile(`^\+\d{0,15}$`)

var phoneNumberSchema = &jsonschema.Schema{
	Types:     []string{"string"},
	Pattern:   phoneNumberRegexp,
	MaxLength: 16,
}

var dateTimeRegexp = regexp.MustCompile(`^\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}Z$`)
var auditInfoIssueRegexp = regexp.MustCompile(`^[A-Z]+-\d+$`)

var dateTimeSchema = &jsonschema.Schema{
	Types:     []string{"string"},
	Pattern:   dateTimeRegexp,
	MaxLength: ^0,
}

var auditInfoIssueSchema = &jsonschema.Schema{
	Types:     []string{"string"},
	Pattern:   auditInfoIssueRegexp,
	MaxLength: 16,
}

var integerSchema = &jsonschema.Schema{
	Types: []string{"integer"},
}

var stringSchema = &jsonschema.Schema{
	Types:     []string{"string"},
	MaxLength: ^0,
}

var idsSchema = &jsonschema.Schema{
	Types:    []string{"array"},
	Items:    stringSchema,
	MaxItems: ^0,
}

var issuesSchema = &jsonschema.Schema{
	Types:    []string{"array"},
	Items:    auditInfoIssueSchema,
	MaxItems: ^0,
}

var changeInfoSchema = &jsonschema.Schema{
	Types: []string{"object"},
	Properties: map[string]*jsonschema.Schema{
		"issue": issuesSchema,
		"comment": {
			Types:     []string{"string"},
			MaxLength: 128,
		},
	},
	Required:      []string{"issue", "comment"},
	MinProperties: 0,
	MaxProperties: 2,
}

var gateIDsSchema = &jsonschema.Schema{
	Types:    []string{"array"},
	Items:    stringSchema,
	MinItems: 1,
	MaxItems: 3,
}

var blockTypeSchema = &jsonschema.Schema{
	Types:     []string{"string"},
	Enum:      []interface{}{"permanent"},
	MaxLength: ^0,
}

var extraSchema = &jsonschema.Schema{
	Types:                []string{"object"},
	AdditionalProperties: stringSchema,
	MinProperties:        0,
	MaxProperties:        ^0,
}

var regionSchema = &jsonschema.Schema{
	Types: []string{"object"},
	Properties: map[string]*jsonschema.Schema{
		"name": stringSchema,
	},
	Required:             []string{"name"},
	MinProperties:        1,
	MaxProperties:        1,
	AdditionalProperties: false,
}

var regionsPrefixSchema = &jsonschema.Schema{
	Types: []string{"object"},
	PatternProperties: map[*regexp.Regexp]*jsonschema.Schema{
		phoneNumberRegexp: regionSchema,
	},
	MinProperties: 0,
	MaxProperties: ^0,
}

var updateRoutesSchema = &jsonschema.Schema{
	Types: []string{"object"},
	Properties: map[string]*jsonschema.Schema{
		"rule_id":     stringSchema,
		"destination": phoneNumberSchema,
		"gates":       gateIDsSchema,
		"weight":      integerSchema,
		"mode":        stringSchema,
	},
	Required:             []string{"rule_id", "destination", "gates", "weight", "mode"},
	MinProperties:        5,
	MaxProperties:        5,
	AdditionalProperties: false,
}

var createRoutesSchema = &jsonschema.Schema{
	Types: []string{"object"},
	Properties: map[string]*jsonschema.Schema{
		"destination": phoneNumberSchema,
		"gates":       gateIDsSchema,
		"weight":      integerSchema,
		"mode":        stringSchema,
	},
	Required:             []string{"destination", "gates", "weight", "mode"},
	MinProperties:        4,
	MaxProperties:        4,
	AdditionalProperties: false,
}

var updateGatesSchema = &jsonschema.Schema{
	Types: []string{"object"},
	Properties: map[string]*jsonschema.Schema{
		"gateid":     stringSchema,
		"aliase":     stringSchema,
		"fromname":   stringSchema,
		"consumer":   stringSchema,
		"contractor": stringSchema,
		"extra":      extraSchema,
	},
	Required:             []string{"gateid", "aliase", "fromname", "consumer", "contractor"},
	MinProperties:        5,
	MaxProperties:        6,
	AdditionalProperties: false,
}

var createGatesSchema = &jsonschema.Schema{
	Types: []string{"object"},
	Properties: map[string]*jsonschema.Schema{
		"aliase":     stringSchema,
		"fromname":   stringSchema,
		"consumer":   stringSchema,
		"contractor": stringSchema,
		"extra":      extraSchema,
	},
	Required:             []string{"aliase", "fromname", "consumer", "contractor"},
	MinProperties:        4,
	MaxProperties:        5,
	AdditionalProperties: false,
}

var updateBlockedPhonesSchema = &jsonschema.Schema{
	Types: []string{"object"},
	Properties: map[string]*jsonschema.Schema{
		"blockid":   stringSchema,
		"phone":     phoneNumberSchema,
		"blocktype": blockTypeSchema,
		"blocktill": dateTimeSchema,
	},
	Required:             []string{"blockid", "phone", "blocktype", "blocktill"},
	MinProperties:        4,
	MaxProperties:        4,
	AdditionalProperties: false,
}

var createBlockedPhonesSchema = &jsonschema.Schema{
	Types: []string{"object"},
	Properties: map[string]*jsonschema.Schema{
		"phone":     phoneNumberSchema,
		"blocktype": blockTypeSchema,
		"blocktill": dateTimeSchema,
	},
	Required:             []string{"phone", "blocktype", "blocktill"},
	MinProperties:        3,
	MaxProperties:        3,
	AdditionalProperties: false,
}

var updateFallbacksSchema = &jsonschema.Schema{
	Types: []string{"object"},
	Properties: map[string]*jsonschema.Schema{
		"id":      stringSchema,
		"srcgate": stringSchema,
		"srcname": stringSchema,
		"dstgate": stringSchema,
		"dstname": stringSchema,
		"order": {
			Types: []string{"number"},
		},
	},
	Required:             []string{"id", "srcgate", "srcname", "dstgate", "dstname", "order"},
	MinProperties:        6,
	MaxProperties:        6,
	AdditionalProperties: false,
}

var createFallbacksSchema = &jsonschema.Schema{
	Types: []string{"object"},
	Properties: map[string]*jsonschema.Schema{
		"srcgate": stringSchema,
		"srcname": stringSchema,
		"dstgate": stringSchema,
		"dstname": stringSchema,
		"order": {
			Types: []string{"number"},
		},
	},
	Required:             []string{"srcgate", "srcname", "dstgate", "dstname", "order"},
	MinProperties:        5,
	MaxProperties:        5,
	AdditionalProperties: false,
}

var updateTemplatesSchema = &jsonschema.Schema{
	Types: []string{"object"},
	Properties: map[string]*jsonschema.Schema{
		"id":                 stringSchema,
		"abc_service":        stringSchema,
		"sender_meta":        stringSchema,
		"fields_description": stringSchema,
	},
	Required:             []string{"id", "abc_service", "sender_meta", "fields_description"},
	MinProperties:        4,
	MaxProperties:        4,
	AdditionalProperties: false,
}

var createTemplatesSchema = &jsonschema.Schema{
	Types: []string{"object"},
	Properties: map[string]*jsonschema.Schema{
		"text":               stringSchema,
		"abc_service":        stringSchema,
		"sender_meta":        stringSchema,
		"fields_description": stringSchema,
	},
	Required:             []string{"text", "abc_service", "sender_meta", "fields_description"},
	MinProperties:        4,
	MaxProperties:        4,
	AdditionalProperties: false,
}

var updateRegionsSchema = &jsonschema.Schema{
	Types: []string{"object"},
	Properties: map[string]*jsonschema.Schema{
		"id":     stringSchema,
		"name":   stringSchema,
		"prefix": phoneNumberSchema,
	},
	Required:             []string{"id", "name", "prefix"},
	MinProperties:        3,
	MaxProperties:        3,
	AdditionalProperties: false,
}

var createRegionsSchema = &jsonschema.Schema{
	Types: []string{"object"},
	Properties: map[string]*jsonschema.Schema{
		"name":   stringSchema,
		"prefix": stringSchema,
	},
	Required:             []string{"name", "prefix"},
	MinProperties:        2,
	MaxProperties:        2,
	AdditionalProperties: false,
}

var setRegionsSchema = jsonschema.Schema{
	Properties: map[string]*jsonschema.Schema{
		"delete": idsSchema,
		"update": {
			Types:    []string{"array"},
			Items:    updateRegionsSchema,
			MaxItems: ^0,
		},
		"create": {
			Types:    []string{"array"},
			Items:    createRegionsSchema,
			MaxItems: ^0,
		},
		"change_info": changeInfoSchema,
	},
	MinProperties:        0,
	MaxProperties:        4,
	AdditionalProperties: false,
}

var setRoutesSchema = jsonschema.Schema{
	Properties: map[string]*jsonschema.Schema{
		"delete": idsSchema,
		"update": {
			Types:    []string{"array"},
			Items:    updateRoutesSchema,
			MaxItems: ^0,
		},
		"create": {
			Types:    []string{"array"},
			Items:    createRoutesSchema,
			MaxItems: ^0,
		},
		"change_info": changeInfoSchema,
	},
	MinProperties:        0,
	MaxProperties:        4,
	AdditionalProperties: false,
}

var setGatesSchema = jsonschema.Schema{
	Properties: map[string]*jsonschema.Schema{
		"delete": idsSchema,
		"update": {
			Types:    []string{"array"},
			Items:    updateGatesSchema,
			MaxItems: ^0,
		},
		"create": {
			Types:    []string{"array"},
			Items:    createGatesSchema,
			MaxItems: ^0,
		},
		"change_info": changeInfoSchema,
	},
	MinProperties:        0,
	MaxProperties:        4,
	AdditionalProperties: false,
}

var setBlockedPhonesSchema = jsonschema.Schema{
	Properties: map[string]*jsonschema.Schema{
		"delete": idsSchema,
		"update": {
			Types:    []string{"array"},
			Items:    updateBlockedPhonesSchema,
			MaxItems: ^0,
		},
		"create": {
			Types:    []string{"array"},
			Items:    createBlockedPhonesSchema,
			MaxItems: ^0,
		},
		"change_info": changeInfoSchema,
	},
	MinProperties:        0,
	MaxProperties:        4,
	AdditionalProperties: false,
}

var setFallbacksSchema = jsonschema.Schema{
	Properties: map[string]*jsonschema.Schema{
		"delete": idsSchema,
		"update": {
			Types:    []string{"array"},
			Items:    updateFallbacksSchema,
			MaxItems: ^0,
		},
		"create": {
			Types:    []string{"array"},
			Items:    createFallbacksSchema,
			MaxItems: ^0,
		},
		"change_info": changeInfoSchema,
	},
	MinProperties:        0,
	MaxProperties:        4,
	AdditionalProperties: false,
}

var setTemplatesSchema = jsonschema.Schema{
	Properties: map[string]*jsonschema.Schema{
		"update": {
			Types:    []string{"array"},
			Items:    updateTemplatesSchema,
			MaxItems: ^0,
		},
		"create": {
			Types:    []string{"array"},
			Items:    createTemplatesSchema,
			MaxItems: ^0,
		},
		"change_info": changeInfoSchema,
	},
	MinProperties:        0,
	MaxProperties:        3,
	AdditionalProperties: false,
}
