package yasmsinternal

import (
	"fmt"
	"net/http"
	"reflect"
	"strings"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/blackbox/httpbb"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/loggers"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/roles"
	"a.yandex-team.ru/passport/infra/daemons/yasms_internal/internal/yasmsinternal/processor"
	"a.yandex-team.ru/passport/shared/golibs/httpdaemon"
	"a.yandex-team.ru/passport/shared/golibs/httpdaemon/httpdtvm"
	"a.yandex-team.ru/passport/shared/golibs/logger"
	"a.yandex-team.ru/passport/shared/golibs/unistat"
)

type YasmsInternal struct {
	tvm       tvm.Client
	bbClient  *httpbb.Client
	unistat   stats
	processor *processor.Processor
	specPath  string

	accessLog       log.Logger
	modificationLog *loggers.TskvLog
}

type ConfigCommon struct {
	AccessLog       logger.Config `json:"access_log"`
	ModificationLog logger.Config `json:"modification_log"`
}

type Config struct {
	Common          ConfigCommon       `json:"common"`
	Tvm             httpdtvm.TvmConfig `json:"tvm"`
	ProcessorConfig processor.Config   `json:"processor"`
	Spec            string             `json:"openapi_spec"`
	BlackBoxConfig  BlackBoxConfig     `json:"blackbox"`
}

type stats struct {
	requestsByPath *unistat.SignalSet
}

type Factory struct{}

func (f *Factory) NewService(config httpdaemon.ServiceConfig) (httpdaemon.Service, error) {
	var cfg Config
	if err := httpdaemon.ParseServiceConfig(config, &cfg); err != nil {
		return nil, err
	}

	accessLog, err := logger.CreateLog(cfg.Common.AccessLog)
	if err != nil {
		return nil, err
	}

	modificationLog, err := loggers.NewTskvLog(cfg.Common.ModificationLog)
	if err != nil {
		return nil, err
	}

	tvmClient, err := httpdtvm.InitTvm(cfg.Tvm)
	if err != nil {
		return nil, err
	}

	bbClient, err := CreateBlackBox(&cfg.BlackBoxConfig, tvmClient, cfg.Tvm.Destinations["blackbox"])
	if err != nil {
		return nil, fmt.Errorf("failed to create blackbox client: %w", err)
	}

	proc, err := processor.NewProcessor(cfg.ProcessorConfig, tvmClient)
	if err != nil {
		return nil, err
	}

	return &YasmsInternal{
		tvm:      tvmClient,
		bbClient: bbClient,
		unistat: stats{
			requestsByPath: unistat.DefaultChunk.CreateSignalSet("in.requests."),
		},
		processor:       proc,
		specPath:        cfg.Spec,
		accessLog:       accessLog,
		modificationLog: modificationLog,
	}, nil
}

func (t *YasmsInternal) GetOptions() *httpdaemon.Options {
	return &httpdaemon.Options{
		StopService: t.Stop,
	}
}

func (t *YasmsInternal) Stop() {
	t.processor.Stop()
}

type BinderWithMultiQueryParam struct{}

func QueryParamSplitter(paramWithMulti interface{}) interface{} {
	switch paramWithMulti := paramWithMulti.(type) {
	case []string:
		if len(paramWithMulti) != 0 {
			result := make([]string, 0)
			for _, elem := range paramWithMulti {
				if strings.Contains(elem, ",") {
					result = append(result, strings.Split(elem, ",")...)
				} else {
					result = append(result, elem)
				}
			}
			return result
		}
		return paramWithMulti
	}
	return paramWithMulti
}

func (*BinderWithMultiQueryParam) Bind(req interface{}, ctx echo.Context) error {
	binder := new(echo.DefaultBinder)
	err := binder.Bind(req, ctx)
	if err != nil {
		return err
	}

	if req == nil || len(ctx.QueryParams()) == 0 {
		return nil
	}

	typ := reflect.TypeOf(req).Elem()
	val := reflect.Indirect(reflect.ValueOf(req))

	for i := 0; i < typ.NumField(); i++ {
		typeField := typ.Field(i)
		queryMultiTag := typeField.Tag.Get("query_multi")
		if queryMultiTag == "true" {
			rearrangedParam := QueryParamSplitter(val.Field(i).Interface())
			val.Field(i).Set(reflect.ValueOf(rearrangedParam))
		} else {
			val.Field(i).Set(reflect.ValueOf(val.Field(i).Interface()))
		}
	}

	return nil
}

func (t *YasmsInternal) AddHandlers(router *echo.Echo) {
	t.unistat.requestsByPath.CreateSignal("/ping")

	router.Binder = &BinderWithMultiQueryParam{}

	router.Pre(
		t.accessLogMiddleware(),
		t.sendReqIDMiddleware(),
	)
	router.Use(t.handleErrorMiddleware())

	router.GET(
		"/ping",
		t.HandlePing(),
	)

	router.GET(
		"/healthcheck",
		t.HandleHealthCheck(),
	)

	router.GET(
		"/openapi-spec",
		t.HandleSpec(),
	)

	router.GET(
		"/1/user_roles",
		t.HandleGetUserRoles(),
		t.tvmServiceAuthMiddleware(),
		t.tvmUserAuthMiddleware(),
		t.checkGrantsMiddleware(roles.ReadAccess, roles.UserRolesHandler, false),
	)

	router.GET(
		"/1/regions",
		t.HandleGetRegionsV1(),
		t.tvmServiceAuthMiddleware(),
		t.tvmUserAuthMiddleware(),
		t.checkGrantsMiddleware(roles.ReadAccess, roles.RegionsHandler, true),
	)

	router.GET(
		"/1/routes",
		t.HandleGetRoutesV1(),
		t.tvmServiceAuthMiddleware(),
		t.tvmUserAuthMiddleware(),
		t.checkGrantsMiddleware(roles.ReadAccess, roles.RoutesHandler, true),
	)

	router.GET(
		"/1/templates",
		t.HandleGetTemplatesV1(),
		t.tvmServiceAuthMiddleware(),
		t.tvmUserAuthMiddleware(),
		t.checkGrantsMiddleware(roles.ReadAccess, roles.TemplatesHandler, true),
	)

	router.GET(
		"/1/routes/enums",
		t.HandleRouteEnumsV1(),
		t.tvmServiceAuthMiddleware(),
		t.tvmUserAuthMiddleware(),
		t.checkGrantsMiddleware(roles.ReadAccess, roles.RoutesHandler, true),
	)

	router.GET(
		"/1/gates",
		t.HandleGetGatesV1(),
		t.tvmServiceAuthMiddleware(),
		t.tvmUserAuthMiddleware(),
		t.checkGrantsMiddleware(roles.ReadAccess, roles.GatesHandler, true),
	)

	router.GET(
		"/1/blockedphones",
		t.HandleGetBlockedPhonesV1(),
		t.tvmServiceAuthMiddleware(),
		t.tvmUserAuthMiddleware(),
		t.checkGrantsMiddleware(roles.ReadAccess, roles.BlockedPhonesHandler, true),
	)

	router.GET(
		"/1/fallbacks",
		t.HandleGetFallbacksV1(),
		t.tvmServiceAuthMiddleware(),
		t.tvmUserAuthMiddleware(),
		t.checkGrantsMiddleware(roles.ReadAccess, roles.FallbacksHandler, true),
	)

	router.PUT(
		"/1/regions",
		t.HandleSetRegionsV1(),
		t.tvmServiceAuthMiddleware(),
		t.tvmUserAuthMiddleware(),
		t.checkGrantsMiddleware(roles.WriteAccess, roles.RegionsHandler, true),
	)

	router.PUT(
		"/1/routes",
		t.HandleSetRoutesV1(),
		t.tvmServiceAuthMiddleware(),
		t.tvmUserAuthMiddleware(),
		t.checkGrantsMiddleware(roles.WriteAccess, roles.RoutesHandler, true),
	)

	router.PUT(
		"/1/gates",
		t.HandleSetGatesV1(),
		t.tvmServiceAuthMiddleware(),
		t.tvmUserAuthMiddleware(),
		t.checkGrantsMiddleware(roles.WriteAccess, roles.GatesHandler, true),
	)

	router.PUT(
		"/1/blockedphones",
		t.HandleSetBlockedPhonesV1(),
		t.tvmServiceAuthMiddleware(),
		t.tvmUserAuthMiddleware(),
		t.checkGrantsMiddleware(roles.WriteAccess, roles.BlockedPhonesHandler, true),
	)

	router.PUT(
		"/1/fallbacks",
		t.HandleSetFallbacksV1(),
		t.tvmServiceAuthMiddleware(),
		t.tvmUserAuthMiddleware(),
		t.checkGrantsMiddleware(roles.WriteAccess, roles.FallbacksHandler, true),
	)

	router.PUT(
		"/1/templates",
		t.HandleSetTemplatesV1(),
		t.tvmServiceAuthMiddleware(),
		t.tvmUserAuthMiddleware(),
		t.checkGrantsMiddleware(roles.WriteAccess, roles.TemplatesHandler, true),
	)

	router.POST(
		"/1/templates_parse",
		t.HandleTemplatesParseV1(),
		t.tvmServiceAuthMiddleware(),
		t.checkGrantsMiddleware(roles.WriteAccess, roles.TemplatesHandler, false),
	)

	router.GET(
		"/1/service/routes",
		t.HandleGetRoutesV1(),
		t.tvmServiceAuthMiddleware(),
		t.checkServiceGrantsMiddleware(roles.ReadAccess, roles.RoutesHandler),
	)

	router.GET(
		"/1/service/gates",
		t.HandleGetGatesV1(),
		t.tvmServiceAuthMiddleware(),
		t.checkServiceGrantsMiddleware(roles.ReadAccess, roles.GatesHandler),
	)

	router.GET(
		"/1/service/blockedphones",
		t.HandleGetBlockedPhonesV1(),
		t.tvmServiceAuthMiddleware(),
		t.checkServiceGrantsMiddleware(roles.ReadAccess, roles.BlockedPhonesHandler),
	)

	router.GET(
		"/1/service/fallbacks",
		t.HandleGetFallbacksV1(),
		t.tvmServiceAuthMiddleware(),
		t.checkServiceGrantsMiddleware(roles.ReadAccess, roles.FallbacksHandler),
	)

	router.GET(
		"/1/service/templates",
		t.HandleGetTemplatesV1(),
		t.tvmServiceAuthMiddleware(),
		t.checkServiceGrantsMiddleware(roles.ReadAccess, roles.TemplatesHandler),
	)

	router.GET(
		"/1/audit/bulk_info",
		t.HandleGetAuditBulkInfoV1(),
		t.tvmServiceAuthMiddleware(),
		t.tvmUserAuthMiddleware(),
		t.checkGrantsMiddleware(roles.ReadAccess, roles.AuditInfoHandler, true),
	)

	router.GET(
		"/1/audit/change_info",
		t.HandleGetAuditChangeInfoV1(),
		t.tvmServiceAuthMiddleware(),
		t.tvmUserAuthMiddleware(),
		t.checkGrantsMiddleware(roles.ReadAccess, roles.AuditInfoHandler, true),
	)
}

func (t *YasmsInternal) HandlePing() echo.HandlerFunc {
	return func(ctx echo.Context) error {
		logger.Log().Debugf("Ping: service is up")
		return ctx.NoContent(http.StatusOK)
	}
}

func (t *YasmsInternal) HandleSpec() echo.HandlerFunc {
	return func(ctx echo.Context) error {
		return ctx.File(t.specPath)
	}
}

type GetUserRolesResponse struct {
	UID   tvm.UID             `json:"uid"`
	Roles roles.UserRolesList `json:"roles"`
}

func (t *YasmsInternal) HandleGetUserRoles() echo.HandlerFunc {
	return func(ctx echo.Context) error {
		r, err := t.tvm.GetRoles(ctx.Request().Context())
		if err != nil {
			return &errs.UnknownError{
				Status:  errs.Error,
				Message: "Failed to check roles",

				Component: errs.CommonComponent,
			}
		}

		ut := tvm.ContextUserTicket(ctx.Request().Context())
		userRoles, err := r.GetRolesForUser(ut, nil)
		if err != nil {
			return &errs.UnauthorizedError{
				Status:  errs.Error,
				Message: fmt.Sprintf("Failed to check user roles: %s", err.Error()),

				Component: errs.CommonComponent,
			}
		}

		return ctx.JSON(http.StatusOK, &GetUserRolesResponse{
			UID:   ut.DefaultUID,
			Roles: roles.ConvertUserRoles(userRoles),
		})
	}
}

func (t *YasmsInternal) ModificationLog(ctx echo.Context) *loggers.TskvContextLog {
	return t.modificationLog.WithContext(ctx.Request().Context())
}
