# -*- coding: utf-8 -*-
import logging
from logging.config import dictConfig

from flask import Flask
from passport.backend.core.grants import get_grants_config
from passport.backend.core.host.host import get_current_host
from passport.backend.core.tvm import get_tvm_credentials_manager
from passport.backend.library.wsgi_runner import Runner
from passport.infra.daemons.yasmsapi.api import (
    logging_utils,
    views,
)
from passport.infra.daemons.yasmsapi.api.configs import config
from passport.infra.daemons.yasmsapi.api.loggers import (
    APP_LOGGER_NAME,
    LOGGING,
)
from passport.infra.daemons.yasmsapi.db.config import DB_NAMES
from passport.infra.daemons.yasmsapi.db.connection import (
    DBError,
    get_db_connection,
)
from passport.infra.daemons.yasmsapi.api.sms_encryptor import get_sms_encryptor


log = logging.getLogger('yasms.initialize')


def create_app():
    app = Flask(__name__)
    # Хотим со слэшами в конце и без
    app.url_map.strict_slashes = False
    app.add_url_rule('/ping', view_func=views.ping, methods=['GET'])
    app.add_url_rule(
        '/sendsms',
        view_func=views.SendSmsView.as_view('sendsms_main'),
        methods=['GET', 'POST'],
    )
    app.add_url_rule(
        '/2/sendsms',
        view_func=views.SendSmsView.as_view('sendsms_test'),
        methods=['GET', 'POST'],
    )
    app.add_url_rule(
        '/routing',
        view_func=views.RoutingView.as_view(),
        methods=['GET'],
    )
    app.before_request(logging_utils.prepare_request_env)
    app.before_request(logging_utils.setup_log_prefix)
    app.before_request(logging_utils.log_request)

    app.after_request(logging_utils.log_response)
    return app


def run_with_logging(func, name):
    log.info(u'Start initializing %s', name)
    try:
        func()
    except Exception as e:
        log.info(u'Stop initializing %s, status ERROR', name)
        log.error(u'Failed with error %s', unicode(e))
        raise
    log.info(u'Stop initializing %s, status OK', name)


def prepare_environment():
    # настраиваем логгинг
    dictConfig(LOGGING)

    # запрашиваем хост при запуске приложения, чтобы быть уверенными
    # что этот хост есть в конфигах, иначе будет RuntimeError
    get_current_host()

    get_grants_config()

    get_tvm_credentials_manager()

    get_sms_encryptor()


def prepare_db_connections():
    # инициализируем пул соединений с базой
    db_connection = get_db_connection()
    run_with_logging(
        lambda: db_connection.configure(),
        'Database Configuration',
    )
    for db_name in DB_NAMES:
        engine = get_db_connection().get_engine(db_name)
        try:
            conn = engine.connect()
            # Делаем запрос, чтобы Алхимия реально открыла соединение с СУБД
            # https://st.yandex-team.ru/PASSP-19473
            conn.execute('select 1')
            # close возвращает соединение в пул Алхимии
            conn.close()
        except DBError as e:
            log.warning(u'Failed to create db connection: %s', unicode(e))


def execute_app():
    # настраиваем settings
    prepare_environment()
    application = create_app()
    application.config['LOGGER_NAME'] = APP_LOGGER_NAME

    return application


def post_worker_init(worker):
    prepare_db_connections()


def main():
    config.set_as_passport_settings()
    config.set_logging()
    app = execute_app()
    Runner(app, config).run()
