# -*- coding: utf-8 -*-
import re

from passport.backend.core import validators
from passport.backend.core.builders.yasms.utils import normalize_phone_number
from passport.infra.daemons.yasmsapi.common.helpers import count_sms_segments


DEFAULT_ROUTE = 'default'
MAX_SMS_SEGMENTS = 10

_normalized_phone_number_re = re.compile(ur'^\+\d{8,20}$')


def form_encode_invalid_to_field_and_code(exception):
    # Перл возвращает ошибки по одной, а не все сразу
    field = exception.error_dict.keys()[0]
    code = exception.error_dict[field].code
    return field, code


class SmsTextValidator(validators.FancyValidator):
    strip = True
    not_empty = True

    messages = {
        u'tooLarge': u'Text is too large',
    }

    def validate_python(self, value, state):
        segments_count = count_sms_segments(value)
        if segments_count > MAX_SMS_SEGMENTS:
            raise validators.Invalid(self.message(u'tooLarge', state), value, state)


class RouteValidator(validators.FancyValidator):
    strip = True
    not_empty = True
    if_missing = DEFAULT_ROUTE
    if_invalid = DEFAULT_ROUTE


class NormalizablePhoneNumberValidator(validators.FancyValidator):
    strip = True
    not_empty = True

    messages = {
        u'badPhone': u'Phone number is in wrong format',
    }

    def __init__(self, *args, **kwargs):
        super(NormalizablePhoneNumberValidator, self).__init__(*args, **kwargs)

    def _to_python(self, value, state=None):
        normalized = normalize_phone_number(value)
        if _normalized_phone_number_re.match(normalized) is None:
            raise validators.Invalid(self.message(u'badPhone', state), value, state)
        return normalized
