# -*- coding: utf-8 -*-
import logging
import random
import string
import time

from flask import (
    has_request_context,
    request,
)
from passport.backend.core.host.host import get_current_host
from passport.backend.core.logging_utils.helpers import (
    mask_sensitive_fields,
    trim_message,
)
from passport.backend.core.logging_utils.request_id import RequestIdManager
from passport.infra.daemons.yasmsapi.api.views.base import BaseResponse


log = logging.getLogger('yasms.common')


REQUEST_ID_LAST_CHARS_COUNT = 16
REQUEST_ID_ALPHABET = string.digits + string.ascii_letters
REQUEST_ID_LENGTH = 16


def _generate_request_id(length):
    # g - generated, чтобы было сразу понятно, что request id не прилетел снаружи
    prefix = 'g-'
    return prefix + ''.join(random.choice(REQUEST_ID_ALPHABET) for _ in xrange(length - len(prefix)))


def prepare_request_env():
    # Сохраняем текущее время в качестве времени начала выполнения запроса
    request.start_time = time.time()
    request_id = request.environ.get(
        'HTTP_YA_CLIENT_X_REQUEST_ID',
        request.environ.get('HTTP_X_REQUEST_ID', None),
    )
    if not request_id:
        request_id = _generate_request_id(REQUEST_ID_LENGTH)
    request.request_id = request_id


def get_external_request_id():
    if has_request_context() and hasattr(request, 'request_id'):
        request_id = request.request_id
        if request_id:
            return request_id[-REQUEST_ID_LAST_CHARS_COUNT:]
    return '-'


def setup_log_prefix():
    clear_log_prefix()

    request_id = (
        ('%x' % get_current_host().get_id()).upper(),
        request.start_time,
        get_external_request_id(),
    )
    RequestIdManager.push_request_id(*request_id)


def clear_log_prefix():
    RequestIdManager.clear_request_id()


def log_request():
    sensitive = ['text', 'phone', 'number', 'text_template_params']
    log.info(u'Request started: %s %s', request.method, request.path)
    log.info(
        u'Request all values: %s',
        trim_message(unicode(mask_sensitive_fields(request.values.to_dict(), fields=sensitive))),
    )


def log_response(response):
    if isinstance(response, BaseResponse):
        response_data = response.log_data()
    else:
        response_data = response.data

    response_data = trim_message(response_data)

    log.info(u'Response status: %s', response.status)
    log.info(u'Response sent: %s', response_data)

    clear_log_prefix()

    return response
