# -*- coding: utf-8 -*-
import logging

from passport.backend.core.dynamic_config import (
    BaseDynamicConfig,
    LoadConfigsError,
    load_json,
)
from passport.infra.daemons.yasmsapi.api.configs import config
from passport.backend.core.lazy_loader import (
    lazy_loadable,
    LazyLoader,
)

from cryptography.hazmat.backends import default_backend
from cryptography.hazmat.primitives.ciphers import (
    algorithms,
    Cipher,
    modes,
)
from os import urandom

log = logging.getLogger('yasms.common')


@lazy_loadable()
class SmsEncryptor(BaseDynamicConfig):
    def __init__(self):
        keys_path = config['sms_encryptor']['keys_path']
        cache_time = config['sms_encryptor']['reload_period']

        super(SmsEncryptor, self).__init__(
            config_filenames=[keys_path],
            cache_time=cache_time,
        )

        self.key_depth = config['sms_encryptor']['key_depth']
        self.encryption_on = config['sms_encryptor']['encrypt_sms']

        log.debug(
            "sms encryptor initialized, keys path %s, reload period %d, key depth %d, encryption is %s"
            % (keys_path, cache_time, self.key_depth, "ON" if self.encryption_on else "OFF")
        )

    def read_config_file(self, filename):
        try:
            keys_json = load_json(filename)

            keys_json.sort(key=lambda k: k['create_ts'])

            idx = len(keys_json) - self.key_depth
            if idx < 0:
                idx = 0

            self.key_id = keys_json[idx]['id']
            self.encryption_key = keys_json[idx]['body'].decode('hex')

            log.debug("keys updated, default encryption key id %s" % self.key_id)
        except (IOError, ValueError, KeyError) as e:
            raise LoadConfigsError(e)

    def encrypt_sms(self, text, phone):
        if not self.encryption_on:
            return text

        iv = urandom(12)

        encryptor = Cipher(
            algorithms.AES(self.encryption_key),
            modes.GCM(iv),
            backend=default_backend(),
        ).encryptor()

        encryptor.authenticate_additional_data(phone.encode('utf-8'))

        ciphertext = encryptor.update(text.encode('utf-8')) + encryptor.finalize()

        result = '1:%s:%s:%s:%s' % (
            self.key_id,
            iv.encode('hex'),
            ciphertext.encode('hex'),
            encryptor.tag.encode('hex'),
        )

        return result


def get_sms_encryptor():
    encryptor = LazyLoader.get_instance("SmsEncryptor")
    encryptor.load()
    return encryptor
