# -*- coding: utf-8 -*-
import json
import logging

from flask import (
    request,
    Response,
)
from flask.views import View
from netaddr import (
    AddrFormatError,
    IPAddress,
)
from passport.backend.core import validators
from passport.backend.core.grants import (
    InvalidSourceError,
    MissingRequiredGrantsError,
    MissingTicketError,
    TicketParsingError,
    UnknownConsumerError,
)
from passport.backend.core.grants.grants import get_grants
from passport.infra.daemons.yasmsapi.api import exceptions as errors
from passport.infra.daemons.yasmsapi.api.forms import form_encode_invalid_to_field_and_code
from passport.infra.daemons.yasmsapi.api.grants import get_name_for_checking_grants
from werkzeug.http import Headers


log = logging.getLogger('passport.infra.daemons.yasmsapi.api.views')

INTERNAL_ERROR_CODE = 'INTERROR'
INTERNAL_ERROR_MESSAGE = 'Internal error'


class BaseResponse(Response):
    charset = 'utf-8'
    default_status = 200
    headers = Headers(
        [
            ('Pragma', 'no-cache'),
            ('Cache-control', 'private, max-age=0, no-cache, no-store, must-revalidate'),
            ('Expires', 'Thu, 01 Jan 1970 00:00:00 GMT'),
        ],
    )
    _log_data = {}

    def log_data(self):
        return json.dumps(self._log_data)


class JsonResponse(BaseResponse):
    default_mimetype = 'application/json'

    @classmethod
    def error_response(cls, exc_code, exc_message=None, status_code=None):
        status_code = status_code or cls.default_status
        data = {
            'status': 'error',
            'error': exc_code,
            'message': exc_message,
        }
        cls._log_data = data.copy()
        return cls(
            json.dumps(data),
            headers=cls.headers,
            status=status_code,
        )

    @classmethod
    def success_response(cls, data, status_code=None):
        status_code = status_code or cls.default_status
        data.update(status='ok')
        cls._log_data = data.copy()
        return cls(
            json.dumps(data),
            headers=cls.headers,
            status=status_code,
        )


class XmlResponse(BaseResponse):
    default_mimetype = 'text/xml'
    headers = Headers(
        [
            ('Pragma', 'no-cache'),
            ('Cache-control', 'private, max-age=0, no-cache, no-store, must-revalidate'),
            ('Expires', 'Thu, 01 Jan 1970 00:00:00 GMT'),
            ('Content-Type', 'text/xml; charset=utf-8'),
        ],
    )

    @classmethod
    def error_response(cls, code, message=None, status_code=None):
        status_code = status_code or cls.default_status
        cls._log_data = {
            'status': 'error',
            'error': code,
        }
        return cls(
            cls.format_error_response(message, code),
            headers=cls.headers,
            status=status_code,
        )

    @classmethod
    def success_response(cls, formatted_response, log_data=None, status_code=None):
        status_code = status_code or cls.default_status
        cls._log_data = log_data or {}
        return cls(
            formatted_response,
            headers=cls.headers,
            status=status_code,
        )

    @classmethod
    def format_error_response(cls, message, code):
        return u"""<?xml version="1.0" encoding="utf-8"?>
           <doc>
           <error>{message}</error>
           <errorcode>{code}</errorcode>
           </doc>
        """.format(
            message=message, code=code
        )


class BaseView(View):
    basic_form = None

    form_field_and_code_to_error = {}

    def __init__(self):
        self.required_grants = []
        self.form_values = {}

        self.request = request
        self.sender_ip = None

        self.response_values = {}

    @classmethod
    def as_view(cls, name=None, *args, **kwargs):
        name = name or cls.__name__
        return super(BaseView, cls).as_view(name, *args, **kwargs)

    @property
    def service_ticket(self):
        return request.environ.get('HTTP_X_YA_SERVICE_TICKET', None)

    def get_consumer_ip(self):
        consumer_ip = self.request.environ.get(
            'HTTP_YA_CONSUMER_REAL_IP',
            self.request.environ.get(
                'HTTP_X_REAL_IP',
                self.request.environ.get('REMOTE_ADDR'),
            ),
        )
        try:
            IPAddress(consumer_ip)
        except (AddrFormatError, ValueError):
            log.debug('Access denied. Strange IP %s passed to IP check' % consumer_ip)
            raise errors.DontKnowYou()

        return consumer_ip

    def update_required_grants(self):
        raise NotImplementedError()  # pragma: no cover

    def require_grants(self):
        consumer = get_name_for_checking_grants(
            self.form_values['sender'],
        )
        self.update_required_grants()
        try:
            get_grants().check_access(
                self.sender_ip,
                [consumer],
                self.required_grants,
                service_ticket=self.service_ticket,
            )
        except UnknownConsumerError:
            raise errors.DontKnowYou()
        except MissingTicketError:
            raise errors.TicketMissing()
        except (TicketParsingError, InvalidSourceError):
            raise errors.TicketInvalid()
        except MissingRequiredGrantsError:
            raise errors.NoRights()

    def _validate_params(self, form, unsafe_params):
        try:
            return form.to_python(unsafe_params)
        except validators.Invalid as e:
            field, code = form_encode_invalid_to_field_and_code(e)
            error = self.form_field_and_code_to_error[(field, code)]
            raise error()

    def process_basic_form(self):
        unsafe_params = self.request.values.to_dict()
        safe_params = self._validate_params(self.basic_form, unsafe_params)
        self.form_values.update(safe_params)

    def dispatch_request(self):
        try:
            self.sender_ip = self.get_consumer_ip()
            self.process_basic_form()
            self.require_grants()
            self.process_request()
            response = self.respond_success()
        except errors.BaseError as e:
            response = self.respond_error(e.code, exc_message=e.message)
        except Exception as e:
            log.exception('Unhandled exception: %s', e)
            response = self.respond_error(INTERNAL_ERROR_CODE, INTERNAL_ERROR_MESSAGE, status_code=500)
        return response

    def process_request(self):
        raise NotImplementedError()  # pragma: no cover

    def respond_success(self):
        raise NotImplementedError()  # pragma: no cover

    def respond_error(self, exc_code, exc_message=None, status_code=None):
        raise NotImplementedError()  # pragma: no cover
