# -*- coding: utf-8 -*-
from datetime import (
    datetime,
    timedelta,
)
import logging
import requests

from flask import (
    request,
    Response,
)
from passport.backend.core.host.host import get_current_host
from passport.backend.utils.file import access
from passport.infra.daemons.yasmsapi.api.configs import config
from passport.infra.daemons.yasmsapi.db import queries
from passport.infra.daemons.yasmsapi.db.config import DB_NAMES
from passport.infra.daemons.yasmsapi.db.connection import (
    DB_EXCEPTIONS,
    get_db_connection,
)


log = logging.getLogger('passport.infra.daemons.yasmsapi.api.views.ping')


def ping():
    # Проверяем доступность файла. Если его нет или он недоступен - значит, мы должны закрыться от балансера
    if not access(config['ping_test_file'], source=config['ping_test_file_source']):
        return Response('Service is down', status=503)

    to_check = [c.strip() for c in request.values.to_dict().get('check', '').strip().split(',')]
    errors = []

    if 'db' in to_check:
        for db_name in DB_NAMES:
            engine = get_db_connection().get_engine(db_name)
            connect_url = engine.url
            try:
                log.debug(u'Checking database availability: "%s" (%s)' % (connect_url.database, connect_url.host))
                if db_name == 'smsqueue':
                    host_name = get_current_host().host.name
                    log.debug(u'Checking daemon heartbeat for %s' % host_name)
                    last_heartbeat = queries.get_heartbeat(host_name)
                    if 'daemon' in to_check and (
                        last_heartbeat is None
                        or (
                            datetime.now() - last_heartbeat
                            >= timedelta(seconds=config['daemon_heartbeat_delta'] * config['daemon_heartbeat_period'])
                        )
                    ):
                        log.error('Daemon is dead for %s; last_beat=%s', host_name, last_heartbeat)
                        errors.append('Daemon is dead for %s' % host_name)
                else:
                    conn = engine.connect()
                    conn.execute('select 1')
                    conn.close()
            except DB_EXCEPTIONS:
                errors.append(
                    u'Database is unavailable: "%s" (%s)'
                    % (
                        connect_url.database,
                        connect_url.host,
                    ),
                )

    if 'daemon' in to_check:
        try:
            daemon_ping = requests.get('http://localhost:{port}/ping'.format(port=config['yasmsd']['port']))
            if daemon_ping.status_code != 200:
                errors.append(
                    'Daemon is down: [%d] %s'
                    % (
                        daemon_ping.status_code,
                        daemon_ping.text,
                    ),
                )
        except requests.exceptions.RequestException as e:
            log.error('Daemon is unavailable: %s' % e)
            errors.append('Daemon is unavailable')

    if errors:
        return Response(
            '\n'.join(errors),
            status=503,
        )

    return Response('Pong\n')
