# -*- coding: utf-8 -*-
import logging

from passport.infra.daemons.yasmsapi.api import exceptions as errors
from passport.infra.daemons.yasmsapi.api.forms import RoutingForm
from passport.infra.daemons.yasmsapi.api.grants import get_grant_name_for_action
from passport.infra.daemons.yasmsapi.api.views.base import (
    BaseView,
    JsonResponse,
)
from passport.infra.daemons.yasmsapi.db.queries import (
    get_all_gates,
    get_all_routes,
    get_possible_routes,
)


log = logging.getLogger('passport.infra.daemons.yasmsapi.api.views.routing')

ROUTE_ACTION = 'route'
DUMP_ACTION = 'dump'


class RoutingView(BaseView):
    basic_form = RoutingForm

    form_field_and_code_to_error = {
        (u'phone', u'empty'): errors.BadPhone,
        (u'phone', u'badPhone'): errors.BadPhone,
        (u'phone', u'invalid'): errors.BadPhone,
        (u'sender', u'empty'): errors.NoSender,
        (u'sender', u'missingValue'): errors.NoSender,
        (u'sender', u'invalid'): errors.DontKnowYou,
    }

    def update_required_grants(self):
        self.required_grants.append(
            get_grant_name_for_action('Routing'),
        )

    def process_request(self):
        action = self.form_values['action']
        number = self.form_values['number']
        route = self.form_values['route']

        if action == ROUTE_ACTION:
            if not (number and route):
                raise errors.NoNumberOrRouteError()

            possible_routes = get_possible_routes(number, route)

            if not possible_routes:
                log.error('No possible routes')
                raise errors.InternalError()

            possible_gates = [r['gateid'] for r in possible_routes]
            self.response_values = {
                'number': number,
                'route': route,
                'possible_gates': possible_gates,
            }

        elif action == DUMP_ACTION:
            gates = get_all_gates()
            if not gates:
                log.error('No gates found')
                raise errors.InternalError()

            routes = get_all_routes()
            if not routes:
                log.error('No routes found')
                raise errors.InternalError()

            self.response_values = {
                'gates': gates,
                'routes': routes,
            }

    def respond_success(self):
        return JsonResponse.success_response(self.response_values)

    def respond_error(self, exc_code, exc_message=None, status_code=None):
        return JsonResponse.error_response(exc_code, exc_message=exc_message, status_code=status_code)
