# -*- coding: utf-8 -*-
import logging

from netaddr import (
    AddrFormatError,
    IPAddress,
)
from passport.backend.core.builders.blackbox.blackbox import (
    BaseBlackboxError,
    Blackbox,
)
from passport.backend.core.builders.kolmogor import (
    BaseKolmogorError,
    Kolmogor,
)
from passport.backend.core.conf import settings
from passport.backend.core.exceptions import UnknownUid
from passport.backend.core.host.host import get_current_host
from passport.backend.core.types.phone_number.phone_number import (
    InvalidPhoneNumber,
    PhoneNumber,
)
from passport.backend.core.utils.blackbox import build_account
from passport.backend.core.utils.decorators import cached_property
from passport.infra.daemons.yasmsapi.api import exceptions as errors
from passport.infra.daemons.yasmsapi.api.configs import config
from passport.infra.daemons.yasmsapi.api.exceptions import DBConnectionError
from passport.infra.daemons.yasmsapi.api.forms import SendSmsForm
from passport.infra.daemons.yasmsapi.api.grants import (
    get_grant_name_for_action,
    SEND_ANONYMOUS_SMS_GRANT,
    SMS_FROM_UID_GRANT,
    SPECIFY_GATE_GRANT,
    SPECIFY_PREVIOUS_GATES_GRANT,
)
from passport.infra.daemons.yasmsapi.api.sms_encryptor import get_sms_encryptor
from passport.infra.daemons.yasmsapi.api.views.base import (
    BaseView,
    XmlResponse,
)
from passport.infra.daemons.yasmsapi.common.helpers import (
    count_sms_segments,
    detect_sms_encoding,
    make_weighted_random_choice,
    mask_for_statbox,
    pack_sms_id,
)
from passport.infra.daemons.yasmsapi.common.renderer import render
from passport.infra.daemons.yasmsapi.common.statbox_loggers import (
    YasmsStatboxPrivateLogger,
    YasmsStatboxPublicLogger,
)
from passport.infra.daemons.yasmsapi.db.connection import DBError
from passport.infra.daemons.yasmsapi.db.queries import (
    check_phone_blocked,
    enqueue_sms,
    get_gate_by_id,
    get_possible_routes,
)


SENDER_MOBILE_YANDEX_RU = 'm.yandex.ru'
VOID_ALIASE = 'devnull'
KOLMOGOR_COUNTER_KEY_BY_PHONE = 'by_phone:{}'
KOLMOGOR_COUNTER_KEY_BY_SENDER_AND_PHONE = 'by_sender:{}:by_phone:{}'
GATES_KEYS = ['gateid', 'gateid2', 'gateid3']


log = logging.getLogger('passport.infra.daemons.yasmsapi.api.views.send_sms')


class SendSmsView(BaseView):
    basic_form = SendSmsForm

    form_field_and_code_to_error = {
        (u'from_uid', u'integer'): errors.BadFromUid,
        (u'from_uid', u'tooLow'): errors.BadFromUid,
        (u'from_uid', u'invalid'): errors.BadFromUid,
        (u'gate_id', u'integer'): errors.NoRoute,
        (u'gate_id', u'invalid'): errors.NoRoute,
        (u'phone', u'empty'): errors.BadPhone,
        (u'phone', u'badPhone'): errors.BadPhone,
        (u'phone', u'invalid'): errors.BadPhone,
        (u'sender', u'empty'): errors.NoSender,
        (u'sender', u'missingValue'): errors.NoSender,
        (u'sender', u'invalid'): errors.DontKnowYou,
        (u'text', u'tooLarge'): errors.TextTooLarge,
        (u'text', u'empty'): errors.NoText,
        (u'text', u'missingValue'): errors.NoText,
        (u'text', u'invalid'): errors.NoText,
        (u'uid', u'empty'): errors.NoUid,
        (u'uid', u'integer'): errors.BlackboxError,
        (u'uid', u'tooLow'): errors.BlackboxError,
        (u'uid', u'invalid'): errors.BlackboxError,
        (u'text_template_params', u'badjson'): errors.InvalidTemplateOrParams,
    }

    @cached_property
    def blackbox(self):
        return Blackbox(use_tvm=True)

    @cached_property
    def kolmogor(self):
        return Kolmogor(use_tvm=True)

    @cached_property
    def host_id(self):
        return get_current_host().get_id()

    def to_statbox(
        self,
        action,
        global_smsid,
        local_smsid,
        sender,
        rule,
        gate,
        text,
        number,
        phone_id=None,
        uid=None,
        identity=None,
        caller=None,
        previous_gates=None,
        user_ip=None,
        user_agent=None,
        consumer_ip=None,
    ):
        public_statbox = YasmsStatboxPublicLogger()
        private_statbox = YasmsStatboxPrivateLogger()

        entries = {
            'action': action,
            'global_smsid': global_smsid,
            'local_smsid': local_smsid,
            'sender': sender.lower().replace('yandex.', ''),
            'rule': rule,
            'gate': gate,
            'chars': len(text),
            'segments': count_sms_segments(text),
            'encoding': detect_sms_encoding(text),
            'masked_number': mask_for_statbox(number),
        }
        if phone_id:
            entries.update({'phoneid': phone_id})
        if uid:
            entries.update({'uid': uid})
        if identity:
            entries.update({'identity': identity})
        if caller:
            entries.update({'caller': caller})
        if previous_gates:
            entries.update({'previous_gates': ','.join(list(map(str, previous_gates)))})
        public_statbox.log(**entries)

        private_statbox.bind(**entries)
        private_statbox.log(
            number=number,
            text=text,
            user_ip=user_ip,
            user_agent=user_agent,
            consumer_ip=consumer_ip,
        )

    def update_required_grants(self):
        """
        Плюсы:
            + Мы можем оторвать грант у нежелательных потребителей
            + Явно задаем его
            + По умолчанию он есть у всех, и мы никого не сломаем
        """
        self.required_grants.append(
            get_grant_name_for_action('Route', action=self.form_values['route']),
        )

        if self.form_values['from_uid']:
            self.required_grants.append(SMS_FROM_UID_GRANT)
        elif self.form_values['phone']:
            self.required_grants.append(SEND_ANONYMOUS_SMS_GRANT)

        if self.form_values['gate_id']:
            self.required_grants.append(SPECIFY_GATE_GRANT)
        if self.form_values['previous_gates']:
            self.required_grants.append(SPECIFY_PREVIOUS_GATES_GRANT)

    def get_counter_key(self, number, sender):
        counter_limit_by_sender = config['anonymous_sms_limit_by_sender'].get(sender)
        if counter_limit_by_sender:
            counter_key = KOLMOGOR_COUNTER_KEY_BY_SENDER_AND_PHONE.format(sender, number)
        else:
            counter_key = KOLMOGOR_COUNTER_KEY_BY_PHONE.format(number)
        return counter_key

    def assert_sms_limits(self, number, sender, previous_gates=None):
        counter_key = self.get_counter_key(number, sender)
        previous_gates = previous_gates or []
        try:
            phone_counters = self.kolmogor.get(
                space=config['kolmogor']['keyspace'],
                keys=[counter_key],
            )
            current_count = phone_counters.get(counter_key, 0)
        except BaseKolmogorError:
            log.warning('Request to Kolmogor failed while getting counters for number %s', number)
            current_count = 0

        sms_limit = config['anonymous_sms_limit_by_sender'].get(
            sender,
            config['anonymous_sms_limit'],
        )
        if sms_limit and current_count >= sms_limit:
            private_statbox = YasmsStatboxPrivateLogger()
            private_statbox.log(
                error='limit_exceeded',
                number=number,
                sender=sender,
                count=current_count,
                previous_gates=','.join(list(map(str, previous_gates))),
            )
            raise errors.LimitExceeded()

    def update_counters(self, number, sender):
        counter_key = self.get_counter_key(number, sender)

        try:
            self.kolmogor.inc(
                space=config['kolmogor']['keyspace'],
                keys=[counter_key],
            )
        except BaseKolmogorError:
            log.warning('Request to Kolmogor failed while incrementing counters for number %s', number)

    def get_current_phone(self, uid, phone_id, number):
        user_info_args = {
            'uid': uid,
            'ip': self.sender_ip,
            'phones': 'bound',
            'attributes': settings.BLACKBOX_PHONE_ATTRIBUTES,
            'need_public_name': False,
        }
        try:
            user_info = self.blackbox.userinfo(**user_info_args)
            account = build_account(user_info)
        except UnknownUid:
            log.warning('Unknown uid=%s' % uid)
            raise errors.UnknownUid()
        except BaseBlackboxError as e:
            log.error('GetUserPhones; Error calling blackbox: %s', e)
            raise errors.BlackboxError()

        if not account.phones.all():
            log.warning('No phones for account')
            return

        phone = None
        if phone_id:
            if account.phones.has_id(phone_id):
                phone = account.phones.by_id(phone_id)
        elif number:
            if account.phones.has_number(number):
                phone = account.phones.by_number(number)
        elif account.phones.default:
            phone = account.phones.default

        return phone

    def get_route_by_number_and_rule(self, number, rule, previous_gates=None):
        """
        :rtype: dict
        """
        possible_routes = get_possible_routes(number, rule)

        if not possible_routes:
            return

        if len(possible_routes) > 1:
            i = make_weighted_random_choice([r.get('weight', 1) for r in possible_routes])
            selected_route = possible_routes[i]
        else:
            selected_route = possible_routes[0]

        if not previous_gates:
            return selected_route

        selected_route_gates = [selected_route[gate_key] for gate_key in GATES_KEYS if selected_route.get(gate_key)]
        if set(selected_route_gates) - set(previous_gates):
            # Выбираем первый гейт, куда еще не сходили
            next_gates = [g for g in selected_route_gates if g not in set(previous_gates)]
            selected_route = get_gate_by_id(next_gates[0])
        else:
            # Выбираем от последнего гейта следующий в списке
            previous_gate_id = selected_route_gates.index(previous_gates[-1])

            if previous_gate_id != len(selected_route_gates) - 1:
                selected_route = get_gate_by_id(selected_route_gates[previous_gate_id + 1])

        return selected_route

    def get_gate(self, gate_id=None, number=None, rule=None, previous_gates=None):
        if gate_id is not None:
            gate = get_gate_by_id(gate_id)
        else:
            gate = self.get_route_by_number_and_rule(number, rule, previous_gates=previous_gates)
        return gate

    def get_metadata(
        self,
        sender,
        caller=None,
        uid=None,
        user_ip=None,
        user_agent=None,
        identity=None,
        scenario=None,
        device_id=None,
        request_path=None,
        masked_text=None,
    ):
        """
        :rtype: dict
        """

        metadata = {
            'service': '{}/{}'.format(sender, caller) if caller else sender,
            'scenario': scenario if scenario else identity,
            'request_path': request_path,
            'uid': uid,
            'ip': user_ip,
            'user_agent': user_agent,
            'device_id': device_id,
            'masked_text': masked_text,
        }

        return {key: value for key, value in metadata.items() if value}

    def process_request(self):
        uid = self.form_values['uid']
        from_uid = self.form_values['from_uid']
        number = self.form_values['phone']
        phone_id = self.form_values['phone_id']
        rule = self.form_values['route']
        sender = self.form_values['sender']
        caller = self.form_values['caller']
        gate_id = self.form_values['gate_id']
        text = self.form_values['text']
        allow_unused_text_params = self.form_values['allow_unused_text_params']
        identity = self.form_values['identity']
        previous_gates = self.form_values['previous_gates']
        scenario = self.form_values['scenario']
        device_id = self.form_values['device_id']
        request_path = self.form_values['request_path']

        user_ip = self.request.headers.get('Ya-Consumer-Client-Ip')
        user_agent = self.request.headers.get('Ya-Client-User-Agent')

        text_template_params = self.form_values.get('text_template_params')

        if user_ip:
            try:
                IPAddress(user_ip)
            except (AddrFormatError, ValueError):
                log.debug('Strange IP \'%s\' passed in header "Ya-Consumer-Client-Ip"' % user_ip)
                raise errors.BadUserIp()

        text, text_masked = self.process_text(
            text,
            text_template_params,
            allow_unused_text_params,
            user_ip=user_ip,
            user_agent=user_agent,
        )

        log.debug(
            '/sendsms: sender=%s caller=%s uid=%s from_uid=%s number=%s phone_id=%s',
            sender or '-',
            caller or '-',
            uid or '-',
            from_uid or '-',
            number or '-',
            phone_id or '-',
        )

        if number:
            # хотим проверить, сколько невалидных номеров по libphonenumbers приходит
            try:
                PhoneNumber.parse(number)
            except InvalidPhoneNumber:
                log.warning(
                    'Invalid phone sender=%s caller=%s uid=%s from_uid=%s number=%s phone_id=%s',
                    sender or '-',
                    caller or '-',
                    uid or '-',
                    from_uid or '-',
                    number or '-',
                    phone_id or '-',
                )

        if from_uid:
            uid = from_uid
        elif uid:
            current_phone = self.get_current_phone(uid, phone_id, number)
            if not current_phone:
                raise errors.NoCurrent()

            phone_id = current_phone.id
            number = current_phone.number.e164

        is_blocked = check_phone_blocked(number)
        if is_blocked:
            raise errors.PhoneBlocked()

        gate = self.get_gate(gate_id=gate_id, number=number, rule=rule, previous_gates=previous_gates)
        if not gate:
            log.warning('Gate not found %s', gate_id)
            raise errors.NoRoute()
        gate_id = gate['gateid']
        gates = (previous_gates or []) + [gate_id]

        if gate['aliase'] == VOID_ALIASE:
            self.response_values.update(
                {
                    'sms_id': pack_sms_id(
                        sms_id=1,
                        host_id=self.host_id,
                    ),
                    'gates': ','.join(list(map(str, gates))),
                }
            )
            return

        self.assert_sms_limits(number, sender, previous_gates)

        try:
            encrypted_masked_text = None
            if text_masked != text:
                encrypted_masked_text = get_sms_encryptor().encrypt_sms(text_masked, number)

            sms_id = enqueue_sms(
                number,
                gate_id,
                get_sms_encryptor().encrypt_sms(text, number),
                sender,
                self.get_metadata(
                    sender=sender,
                    caller=caller,
                    uid=uid,
                    user_ip=user_ip,
                    user_agent=user_agent,
                    identity=identity,
                    scenario=scenario,
                    device_id=device_id,
                    request_path=request_path,
                    masked_text=encrypted_masked_text,
                ),
            )
        except DBError as e:
            raise DBConnectionError(e)

        packed_sms_id = pack_sms_id(
            sms_id=sms_id,
            host_id=self.host_id,
        )

        self.to_statbox(
            action='enqueued',
            global_smsid=packed_sms_id,
            local_smsid=sms_id,
            sender=sender,
            rule=rule,
            gate=gate_id,
            text=text_masked,
            number=number,
            phone_id=phone_id,
            uid=uid,
            caller=caller,
            identity=identity,
            previous_gates=previous_gates,
            user_ip=user_ip,
            user_agent=user_agent,
            consumer_ip=self.sender_ip,
        )
        self.update_counters(number, sender)
        self.response_values.update(
            {
                'sms_id': packed_sms_id,
                'gates': ','.join(list(map(str, gates))[-3:]),
            }
        )

    def process_text(self, template, params, allow_unused_text_params, user_ip='-', user_agent='-'):
        if not params:
            return template, template

        if 'text_template_params' not in self.request.form:
            log.debug('/sendsms template params outside of body. ip: %s, agent: %s', user_ip, user_agent)
            raise errors.UserDataNotInBody()

        try:
            return render(template, params, allow_unused_text_params)
        except ValueError as e:
            log.debug('/sendsms failed to render template: %s. ip: %s, agent: %s', str(e), user_ip, user_agent)
            raise errors.InvalidTemplateOrParams()

    def respond_success(self):
        formatted_response = self.format_xml_success_response()
        log_data = {
            'status': 'ok',
            'sms_id': self.response_values['sms_id'],
            'gates': self.response_values['gates'],
        }
        return XmlResponse.success_response(formatted_response, log_data=log_data)

    def respond_error(self, exc_code, exc_message=None, status_code=None):
        return XmlResponse.error_response(
            exc_code,
            message=exc_message,
            status_code=status_code,
        )

    def format_xml_success_response(self):
        return u"""<?xml version="1.0" encoding="utf-8"?>
           <doc>
           <message-sent id="{sms_id}" />
           <gates ids="{gates}" />
           </doc>
        """.format(
            sms_id=self.response_values['sms_id'], gates=self.response_values['gates']
        )
