# -*- coding: utf-8 -*-
import logging
from math import ceil
import random

import gsm0338  # noqa


log = logging.getLogger('yasms.helpers')


SINGLE_SEGMENT_MAX = {
    'gsm0338': 160,
    'utf16': 70,
}

MULTIPLE_SEGMENTS_MAX = {
    'gsm0338': 153,
    'utf16': 67,
}


def detect_sms_encoding(text):
    """
    Или gsm0338, или UTF-16
    """
    try:
        text.encode('gsm03.38')
    except ValueError:
        return 'utf16'
    return 'gsm0338'


def count_sms_segments(text):
    try:
        text = text.decode('utf8')
    except UnicodeEncodeError:
        pass

    encoding = detect_sms_encoding(text)
    length = len(text)

    if not length:
        return 0

    single_max = SINGLE_SEGMENT_MAX[encoding]
    multiple_max = MULTIPLE_SEGMENTS_MAX[encoding]

    segments = 1

    if length > single_max:
        segments = int(ceil(float(length) / multiple_max))

    return segments


def make_weighted_random_choice(weights_list):
    """
    Принимает список весов, отдает индекс выбранного веса
    :return:
    """
    total_weight = sum(weights_list)
    threshold = random.randint(0, total_weight)
    current_sum = 0

    for i, weight in enumerate(weights_list):
        current_sum += weight

        if threshold <= current_sum:
            return i
    return -1


def pack_sms_id(sms_id, host_id, is_anonymous=True):
    type_send = 2 if is_anonymous else 1
    if not isinstance(sms_id, (int, long)):
        log.error('Bad or missing message id in pack_smsid call')
        return
    if not isinstance(host_id, int):
        log.error('Bad or missing HostID in pack_smsid call')
        return
    if host_id > 999:
        log.error('HostID contains too big value in pack_smsid call')
        return
    packed_sms_id = int('%d%03d000000000000' % (type_send, host_id))
    packed_sms_id += sms_id
    return packed_sms_id


def mask_for_statbox(raw_number, cut_to=4):
    return raw_number[:-cut_to] + '*' * cut_to


def is_string(value):
    return isinstance(value, basestring)


def mask_template_params(value):
    return value if len(value) < 3 else value[0] + '*' * (len(value) - 2) + value[-1]
