# -*- coding: utf-8 -*-

import re

from helpers import mask_template_params, is_string

KEY_PATTERN = re.compile('^\\w+$')


def render(template, params, allow_unused_params=False):
    """
    Replaces template parameters with dict values

    Args:
        template(str): a string containing plain text and parameter names in double braces
        params(dict[basestring]): parameters to be substituted into text
        allow_unused_params(bool): specifies whether parameters not used in the template are allowed or not

    Returns:
        tuple[basestring, basestring]: a tuple of unicode strings (rendered, masked)

    Raises:
        ValueError: if the arguments are invalid.

    Examples:
        render('abc {{ param1 }} def: {{ param2 }}', {'param1': '1', 'param2': 'v_1'})
        ->
        ('abc 1 def: v_1')
    """
    param_pos_start, param_pos_end = 0, 0
    # Pre-allocating assuming each param is used once
    rendered = []
    masked = []

    used_params = set()

    while True:
        param_pos_start = template.find('{{', param_pos_start)

        if param_pos_start == -1:
            rendered.append(template[param_pos_end:])
            masked.append(template[param_pos_end:])
            break

        rendered.append(template[param_pos_end:param_pos_start])
        masked.append(template[param_pos_end:param_pos_start])

        param_pos_start += 2
        param_pos_end = template.find('}}', param_pos_start)

        if param_pos_end == -1:
            raise ValueError('invalid template')

        key = template[param_pos_start:param_pos_end].strip()
        param_pos_end += 2

        if not KEY_PATTERN.match(key):
            raise ValueError('invalid key in template')
        if key not in params:
            raise ValueError('missing key')
        used_params.add(key)

        value = params[key]
        value = value if is_string(value) else str(value)
        rendered.append(value)
        masked.append(mask_template_params(value))
        param_pos_start = param_pos_end

    if not allow_unused_params and len(used_params) != len(params):
        raise ValueError('unused parameters')

    rendered = ''.join(rendered)
    masked = ''.join(masked)

    if '{{' in rendered or '}}' in rendered:
        raise ValueError('special characters outside of template definition')

    return rendered, masked
