# -*- coding: utf-8 -*-
import logging

from frozendict import frozendict
import MySQLdb
from passport.backend.core.lazy_loader import (
    lazy_loadable,
    LazyLoader,
)
from passport.infra.daemons.yasmsapi.db.config import DEFAULT_CONFIGS
from sqlalchemy import create_engine
from sqlalchemy.engine.url import URL
import sqlalchemy.exc
from sqlalchemy.pool import QueuePool


DB_EXCEPTIONS = (
    sqlalchemy.exc.DatabaseError,
    sqlalchemy.exc.InterfaceError,
    MySQLdb.InterfaceError,
    MySQLdb.DatabaseError,
)

log = logging.getLogger('yasms.db')


DB_POOL_SIZE = 3
DB_POOL_MAX_OVERFLOW = 10
DB_POOL_RECYCLE = 600

# параметры подключения к БД
DB_DEFAULT_CONNECT_ARGS = frozendict(
    {
        'mysql': frozendict(),
        'sqlite': {},
    }
)


class DBError(Exception):
    pass


@lazy_loadable()
class DBConnection(object):
    def __init__(self, configs=None):
        """
        configs = {
            'sms': {
                'host': 'cnt-dbm-test.passport.yandex.net',
                'port': 3306,
                'sock': '/mysql.sock'
                'database': 'sms',
                'user': 'sms',
                'driver': 'mysql',
            },
        }
        """
        self.configs = configs or DEFAULT_CONFIGS
        self._engines = {}

    def configure(self):
        for db_name, config in self.configs.iteritems():
            query = {}
            if config.get('sock'):
                query.update(
                    {
                        'unix_socket': config['sock'],
                    }
                )
            if config.get('charset'):
                query.update(
                    {
                        'charset': config['charset'],
                    }
                )
            engine = create_engine(
                URL(
                    drivername=config.get('driver'),
                    username=config.get('user'),
                    password=config.get('password'),
                    host=config.get('host'),
                    port=config.get('port'),
                    database=config.get('database'),
                    query=query,
                ),
                connect_args=DB_DEFAULT_CONNECT_ARGS[config['driver']],
                poolclass=QueuePool,
                pool_size=DB_POOL_SIZE,
                pool_recycle=DB_POOL_RECYCLE,
                max_overflow=DB_POOL_MAX_OVERFLOW,
                echo=False,
            )
            if 'sqlite' in config['driver']:
                engine.raw_connection().connection.text_factory = str
            self._engines.update(
                {
                    db_name: engine,
                }
            )

    def get_engine(self, db_name):
        if not self._engines:
            self.configure()

        engine = self._engines.get(db_name)
        if not engine:
            raise RuntimeError('Can\'t find engine by db_name: %s', db_name)

        return engine

    def execute(self, db_name, query, args=None):
        engine = self.get_engine(db_name)
        try:
            result = engine.execute(query, args or [])
        except DB_EXCEPTIONS as e:
            log.error(e)
            raise DBError(e.message)
        return result


def get_db_connection():
    return LazyLoader.get_instance('DBConnection')
