# -*- coding: utf-8 -*-
from sqlalchemy.schema import (
    Column,
    Index,
    MetaData,
    Table,
)
from sqlalchemy.types import (
    BigInteger,
    DATETIME,
    Enum,
    Integer,
    SmallInteger,
    Text,
    TIMESTAMP,
    VARCHAR,
)


STATUS_READY = 'ready'
STATUS_QUEUED = 'queued'
STATUS_DELIVERED = 'delivered'
STATUS_NOT_DELIVERED = 'notdelivered'
STATUS_SENT = 'senttosmsc'
STATUS_NOT_SENT = 'notsenttosmsc'
STATUS_LOCAL = 'localqueue'

STATUSES = (
    STATUS_READY,
    STATUS_LOCAL,
    STATUS_SENT,
    STATUS_NOT_SENT,
    STATUS_QUEUED,
    STATUS_DELIVERED,
    STATUS_NOT_DELIVERED,
)

DEFAULT_FROM = 'Yandex'
DEFAULT_DELAY = 100
DEFAULT_MODE = 'default'

BLOCKING_TYPE_PERMANENT = 'permanent'
BLOCKING_TYPE_TEMPORARY = 'temporary'
BLOCKING_TYPES = [
    BLOCKING_TYPE_PERMANENT,
    BLOCKING_TYPE_TEMPORARY,
]


queue_metadata = MetaData()
sms_metadata = MetaData()

sms_queue_table = Table(
    'smsqueue_anonym',
    queue_metadata,
    # BigInteger(20)
    Column('smsid', Integer, nullable=False, autoincrement=True, primary_key=True),
    Column('phone', VARCHAR(20), nullable=False, default=''),
    Column('status', Enum(*STATUSES), nullable=False, default=STATUS_READY),
    Column('gateid', BigInteger),
    Column('dlrmessage', Text),
    Column('text', Text),
    Column('create_time', TIMESTAMP, nullable=False, server_default='0000-00-00 00:00:00'),
    Column('touch_time', TIMESTAMP, nullable=False, server_default='0000-00-00 00:00:00'),
    Column('sender', VARCHAR(20)),
    Column('errors', SmallInteger, default=0),
    Column('metadata', Text, default=''),
    Index('queuelookup', 'smsid', 'gateid', 'status', 'create_time', 'touch_time'),
    Index('userqueue', 'smsid', 'phone', 'status', 'create_time', 'touch_time'),
    Index('senders', 'sender'),
    Index('ctime_idx', 'create_time'),
    Index('sendsmslookup', 'status', 'touch_time'),
    sqlite_autoincrement=True,
    mysql_charset='latin1',
)

sms_gates_table = Table(
    'smsgates',
    sms_metadata,
    # BigInteger(20)
    Column('gateid', Integer, nullable=False, autoincrement=True),
    Column('aliase', VARCHAR(length=20), nullable=False, default=''),
    Column('fromname', VARCHAR(length=20), nullable=False, default=DEFAULT_FROM),
    Column('description', VARCHAR(length=60)),
    Column('delay', Integer, nullable=False, default=DEFAULT_DELAY),
    Index('gateseek', 'gateid', 'aliase'),
    sqlite_autoincrement=True,
    mysql_charset='latin1',
)


sms_routes_table = Table(
    'smsrt',
    sms_metadata,
    # BigInteger(20)
    Column('ruleid', Integer, nullable=False, primary_key=True),
    Column('destination', VARCHAR(length=14), nullable=False, primary_key=True),
    Column('gateid', BigInteger, nullable=False, default=0),
    Column('gateid2', BigInteger, nullable=False, default=0),
    Column('gateid3', BigInteger, nullable=False, default=0),
    Column('mode', VARCHAR(length=10), nullable=False, default=DEFAULT_MODE),
    Column('weight', SmallInteger, default=1),
    Column('groupname', VARCHAR(length=40), nullable=False, default=''),
    Index('gateid', 'gateid'),
    Index('bymode', 'mode', 'destination'),
    sqlite_autoincrement=True,
    mysql_charset='latin1',
)


blocked_phones_table = Table(
    'blockedphones',
    sms_metadata,
    # BigInteger(20)
    Column('blockid', Integer, nullable=False, primary_key=True),
    Column('phone', VARCHAR(length=14), nullable=False, primary_key=True, default=''),
    Column('phoneid', BigInteger, nullable=False, primary_key=True, default=0),
    Column('blocktype', Enum(*BLOCKING_TYPES), nullable=False, primary_key=True, default=BLOCKING_TYPE_TEMPORARY),
    Column('blocktill', DATETIME, nullable=False, primary_key=True, server_default='0000-00-00 00:00:00'),
    sqlite_autoincrement=True,
    mysql_charset='latin1',
)


daemon_heartbeat_table = Table(
    'daemon_heartbeat',
    queue_metadata,
    Column('hostname', VARCHAR(length=255), nullable=False, primary_key=True),
    Column('beat_time', TIMESTAMP, nullable=False, server_default='0000-00-00 00:00:00'),
)


BLOCKED_PHONES_TABLE = 'blockedphones'
DAEMON_HEARTBEAT_TABLE = 'daemon_heartbeat'
SMS_GATES_TABLE = 'smsgates'
SMS_QUEUE_TABLE = 'smsqueue_anonym'
SMS_ROUTES_TABLE = 'smsrt'


TABLE_NAME_TO_TABLE = {
    BLOCKED_PHONES_TABLE: blocked_phones_table,
    DAEMON_HEARTBEAT_TABLE: daemon_heartbeat_table,
    SMS_GATES_TABLE: sms_gates_table,
    SMS_QUEUE_TABLE: sms_queue_table,
    SMS_ROUTES_TABLE: sms_routes_table,
}
