package fraud

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"strings"
	"time"

	"github.com/go-resty/resty/v2"
	"github.com/gofrs/uuid"

	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/shared/golibs/utils"
)

const TvmAntiFraudAlias string = "antifraud"
const HeaderServiceTicket string = "X-Ya-Service-Ticket"

const (
	AntiFraudChannel                 string = "pharma"
	AntiFraudSubChannel              string = "yasms"
	AntiFraudPhoneConfirmationMethod string = "by_sms"
)

type AntiFraudConfig struct {
	Host           string         `json:"host"`
	Port           uint16         `json:"port"`
	RequestTimeout utils.Duration `json:"timeout"`
}

type AntiFraudChecker struct {
	host           string
	tvmClient      tvm.Client
	httpClient     *resty.Client
	externalID     uuid.Generator
	requestTimeout time.Duration
}

type AntiFraudRequest struct {
	ExternalID              string `json:"external_id"`
	Timestamp               int64  `json:"t"`
	Channel                 string `json:"channel"`
	SubChannel              string `json:"sub_channel"`
	UserPhone               string `json:"user_phone"`
	PhoneConfirmationMethod string `json:"phone_confirmation_method"`
	Service                 string `json:"service,omitempty"`
	RequestPath             string `json:"request_path,omitempty"`
	Scenario                string `json:"scenario,omitempty"`
	UID                     uint64 `json:"uid,omitempty"`
	UserIP                  string `json:"ip,omitempty"`
	UserAgent               string `json:"user_agent,omitempty"`
	DeviceID                string `json:"device_id,omitempty"`
	MaskedText              string `json:"masked_text"`
	AttemptNo               uint64 `json:"attempt_no"`
}

func (request AntiFraudRequest) String() string {
	return fmt.Sprintf("id: %s, time: %d, phone: %s, service: %s, scenario: %s, path: %s, ip: %s, uid: %d, ua: %s, device_id: %s",
		request.ExternalID, request.Timestamp, request.UserPhone, request.Service, request.Scenario, request.RequestPath,
		request.UserIP, request.UID, request.UserAgent, request.DeviceID)
}

type AntiFraudResponse struct {
	Action AntiFraudAction `json:"action"`
	Reason string          `json:"reason"`
	Status string          `json:"status"`
	TxID   string          `json:"tx_id"`
	Tags   []interface{}   `json:"tags"`
}

func (response AntiFraudResponse) String() string {
	return fmt.Sprintf("tx_id: %s, action: %s, reason: %s, status: %s,", response.TxID, response.Action, response.Reason, response.Status)
}

func prettifyReason(reason string, sizeBound int) string {
	reason = strings.ReplaceAll(reason, "\n", " ")
	if sizeBound >= len(reason) {
		return reason
	}
	reason = reason[:(sizeBound + 1)]
	if cutFrom := strings.LastIndex(reason, " "); cutFrom != -1 {
		return reason[:cutFrom]
	}
	return ""
}

func parseAntiFraudRetry(tags []interface{}) *AntiFraudRetry {
	for _, tag := range tags {
		taggedMap, isMap := tag.(map[string]interface{})
		if !isMap {
			continue
		}

		tagType, ok := taggedMap["type"]
		if !ok || tagType != "retry" {
			continue
		}

		retryData, ok := taggedMap["data"]
		if !ok {
			continue
		}

		data, ok := retryData.(map[string]interface{})
		if !ok {
			continue
		}

		retryDelay, ok := data["delay"]
		if !ok {
			continue
		}

		fdelay, ok := retryDelay.(float64)
		if !ok {
			continue
		}

		retryCount, ok := data["count"]
		if !ok {
			continue
		}

		fcount, ok := retryCount.(float64)
		if !ok {
			continue
		}

		return &AntiFraudRetry{
			Delay: time.Duration(fdelay) * time.Millisecond,
			Count: uint64(fcount),
		}
	}

	return nil
}

// Неожиданный код ответа HTTP.
type UnexpectedHTTPCodeError struct {
	ActualStatusCode   int    // Актуальный код.
	ExpectedStatusCode int    // Ожидаемый код.
	Message            string // Содержимое ответа.
}

// error interface
func (e *UnexpectedHTTPCodeError) Error() string {
	message := fmt.Sprintf("unexpected http code - expected: %d, actual: %d", e.ExpectedStatusCode, e.ActualStatusCode)
	if len(e.Message) > 0 {
		message = fmt.Sprintf("%s, response: '%s'", message, e.Message)
	}

	return message
}

func NewAntiFraudChecker(config *AntiFraudConfig, tvmClient tvm.Client) (*AntiFraudChecker, error) {
	httpClient := resty.New()
	httpClient.SetBaseURL(fmt.Sprintf("%s:%d", config.Host, config.Port))

	if config.RequestTimeout.Duration == 0 {
		config.RequestTimeout.Duration = 500 * time.Millisecond
	}

	return &AntiFraudChecker{
		host:           config.Host,
		tvmClient:      tvmClient,
		httpClient:     httpClient,
		externalID:     uuid.NewGen(),
		requestTimeout: config.RequestTimeout.Duration,
	}, nil
}

func makeRequest(externalID uuid.UUID, metadata Metadata) AntiFraudRequest {
	ip := metadata.UserIP.String()
	if len(metadata.UserIP) == 0 {
		ip = ""
	}

	return AntiFraudRequest{
		ExternalID:              fmt.Sprintf("track-%s", externalID),
		Channel:                 AntiFraudChannel,
		SubChannel:              AntiFraudSubChannel,
		PhoneConfirmationMethod: AntiFraudPhoneConfirmationMethod,

		Timestamp:   metadata.Timestamp.UnixMilli(),
		Service:     metadata.Service,
		Scenario:    metadata.Scenario,
		UserPhone:   metadata.UserPhone,
		RequestPath: metadata.RequestPath,
		UID:         metadata.UID,
		UserIP:      ip,
		UserAgent:   metadata.UserAgent,
		DeviceID:    metadata.DeviceID,
		MaskedText:  metadata.MaskedText,
		AttemptNo:   metadata.AttemptNo,
	}
}

func (checker AntiFraudChecker) Host() string {
	return checker.host
}

func (checker AntiFraudChecker) CheckFraudStatus(metadata Metadata) (*AntiFraudResponse, *AntiFraudRetry, error) {
	ticket, err := checker.tvmClient.GetServiceTicketForAlias(context.Background(), TvmAntiFraudAlias)
	if err != nil {
		return nil, nil, fmt.Errorf("failed to fetch service ticket: %s", err.Error())
	}

	externalID, err := checker.externalID.NewV4()
	if err != nil {
		return nil, nil, fmt.Errorf("failed to generate uuid: %s", err.Error())
	}

	request := makeRequest(externalID, metadata)
	ctx, cancel := context.WithTimeout(context.Background(), checker.requestTimeout)
	defer cancel()

	httpResponse, err := checker.httpClient.R().
		SetContext(ctx).
		SetHeader(HeaderServiceTicket, ticket).
		SetBody(request).
		Post("/score")
	if err != nil {
		// err возвращается as-is чтобы не потерять информацию о типе ошибки (net.Error например) для graphite-логов
		return nil, nil, err
	}

	if httpResponse.StatusCode() != http.StatusOK {
		return nil, nil, &UnexpectedHTTPCodeError{
			ExpectedStatusCode: http.StatusOK,
			ActualStatusCode:   httpResponse.StatusCode(),
			Message:            fmt.Sprintf("%s, (%s)", string(httpResponse.Body()), request.String()),
		}
	}

	response := &AntiFraudResponse{}
	err = json.Unmarshal(httpResponse.Body(), response)
	if err != nil {
		return nil, nil, fmt.Errorf(
			"invalid response body: [%d] %s, (%s)",
			httpResponse.StatusCode(), string(httpResponse.Body()), request.String())
	}

	response.Reason = prettifyReason(response.Reason, 1024)

	if response.Action != AntiFraudActionAllow && response.Action != AntiFraudActionDeny {
		return nil, nil, fmt.Errorf(
			"invalid action: [%d] %s, (%s)",
			httpResponse.StatusCode(), response.String(), request.String())
	}

	return response, parseAntiFraudRetry(response.Tags), nil
}
