package kannel

import (
	"context"
	"math/rand"
	"net/http"
	"sort"
	"sync"
	"time"

	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/logs"
	"a.yandex-team.ru/passport/shared/golibs/utils"
)

/*
Service discovery хоста kannel по имени гейта.

Опрашивает состояние хостов kannel и гейтов на них и обеспечивает балансировку.
*/

type Config struct {
	Hosts          []string       `json:"hosts"`
	PingTimeout    utils.Duration `json:"ping_timeout"`
	StatusTimeout  utils.Duration `json:"status_timeout"`
	PingInterval   utils.Duration `json:"ping_interval"`
	StatusInterval utils.Duration `json:"status_interval"`

	QueueSizeThreshold uint64 `json:"queue_size_threshold"`
}

// Информация о состоянии хоста kannel.
type Monitor struct {
	sync.RWMutex
	host   string        // Имя хоста kannel.
	fqdn   string        // FQDN хоста kannel.
	ping   bool          // Последнее состояние пингера.
	status *KannelStatus // Последнее значение статуса (immutable).
	logs   *logs.Logs    // Логи.
	config *Config
}

// Kannel service discovery.
type Discovery struct {
	config  *Config
	kannels []*Monitor // Описатели состояния kannel.
	logs    *logs.Logs // Логи.
}

// Безопасное обновление пингера kannel.
func (monitor *Monitor) updatePing(ping bool) {
	monitor.Lock()
	defer monitor.Unlock()

	monitor.ping = ping
}

// Безопасное обновление статуса kannel.
func (monitor *Monitor) updateStatus(status *KannelStatus) {
	monitor.Lock()
	defer monitor.Unlock()

	monitor.status = status
}

// Безопасное клонирование значения пингера и статуса kannel для чтения.
func (monitor *Monitor) pingStatus() (bool, *KannelStatus) {
	monitor.RLock()
	defer monitor.RUnlock()

	ping := monitor.ping
	status := monitor.status

	return ping, status
}

// Создание нового маршрутизатора из списка хостов kannel.
func NewKannelDiscovery(config *Config, loggers *logs.Logs) *Discovery {
	result := &Discovery{
		config:  config,
		kannels: make([]*Monitor, len(config.Hosts)),
		logs:    loggers,
	}

	for i, host := range config.Hosts {
		result.kannels[i] = &Monitor{
			config: config,
			host:   host,
			fqdn:   FqdnFromURL(host),
			logs:   loggers,
		}
	}

	return result
}

// Абсолютная разница значений для uint64.
func absDiff(n1, n2 uint64) uint64 {
	if n1 > n2 {
		return n1 - n2
	}
	return n2 - n1
}

// Поиск хоста kannel по имени гейта.
func (service *Discovery) Discovery(smsc string) string {
	// поиск доступных гейтов, которые потенциально могут обработать запрос
	gates := make([]*KannelGateStatus, 0)

	for _, entry := range service.kannels {
		ping, status := entry.pingStatus()

		if !ping || status == nil || !status.online {
			// kannel отключен или недоступен его статус
			continue
		}

		gate, exists := status.gates[smsc]
		if !exists || !gate.online {
			// гейт недоступен или на текущем kannel нет нужного гейта
			continue
		}

		gates = append(gates, gate)
	}

	glen := len(gates)

	// балансировка гейтов
	if glen == 0 {
		// все лежит :(
		return ""
	} else if glen == 1 {
		// при единственном доступном гейте нечего балансировать
		return gates[0].kannel.host
	}

	// PASSPADMIN-4394 (вместо PASSP-22553, с учетом PASSPINCIDENTS-34)
	// Балансировка нагрузки между kannel с учетом размеров очередей.
	//
	// Если разница в размерах очередей между двумя вариантами не превышает
	// значения KannelQueueSizeThreshold (т.е. соответствующая очередь
	// приблизительно равномерно наполняется или опустошается у обоих),
	// то делается случайный выбор.

	// случайное число для выбора среди равноправных гейтов
	random := rand.Int63()

	sort.Slice(gates, func(i, j int) bool {
		// очередь kannel на hdd
		if absDiff(gates[i].kannel.stored, gates[j].kannel.stored) < service.config.QueueSizeThreshold {
			// очередь kannel в ram
			if absDiff(gates[i].kannel.queued, gates[j].kannel.queued) < service.config.QueueSizeThreshold {
				// очередь в гейт
				if absDiff(gates[i].queued, gates[j].queued) < service.config.QueueSizeThreshold {
					// оба гейта находятся в приблизительно одинаковом состоянии (штатный режим)
					return (gates[i].seed ^ random) < (gates[j].seed ^ random)
				}
				return gates[i].queued < gates[j].queued
			}
			return gates[i].kannel.queued < gates[j].kannel.queued
		}
		return gates[i].kannel.stored < gates[j].kannel.stored
	})

	return gates[0].kannel.host
}

// Разовое выполнение пингера одного kannel.
func (monitor *Monitor) pingOnce(ctx context.Context) {
	monitor.logs.General.WriteDebug(logs.ComponentPing, "started: %s", monitor.fqdn)

	timeout, cancel := context.WithTimeout(ctx, monitor.config.PingTimeout.Duration)
	defer cancel()

	record := monitor.logs.Graphite.NewHTTPRecord(monitor.fqdn, logs.ServiceKannel, logs.ActionKannelPing, http.StatusOK, 0)
	value, err := kannelPing(timeout, nil, monitor.host)
	record.Close(err)

	monitor.updatePing(value)

	if err != nil {
		monitor.logs.General.WriteError(logs.ComponentPing, err.Error())
	} else {
		monitor.logs.General.WriteDebug(logs.ComponentPing, "completed: %s", monitor.fqdn)
	}
}

// Монитор пингера одного kannel.
func (monitor *Monitor) pingMonitor(ctx context.Context, wg *sync.WaitGroup) {
	monitor.logs.General.WriteDebug(logs.ComponentPing, "monitor started: %s", monitor.host)

	ticker := time.NewTicker(monitor.config.PingInterval.Duration)

LOOP:
	for {
		monitor.pingOnce(ctx)

		select {
		case <-ticker.C:
			continue
		case <-ctx.Done():
			break LOOP
		}
	}

	ticker.Stop()

	monitor.logs.General.WriteDebug(logs.ComponentPing, "monitor stopped: %s", monitor.host)

	wg.Done()
}

// Разовое выполнение обновления статуса одного kannel.
func (monitor *Monitor) statusOnce(ctx context.Context) {
	monitor.logs.General.WriteDebug(logs.ComponentStatus, "started: %s", monitor.fqdn)

	timeout, cancel := context.WithTimeout(ctx, monitor.config.StatusTimeout.Duration)
	defer cancel()

	record := monitor.logs.Graphite.NewHTTPRecord(monitor.fqdn, logs.ServiceKannel, logs.ActionKannelStatus, http.StatusOK, 0)
	value, err := kannelStatus(timeout, nil, monitor.host)
	record.Close(err)

	monitor.updateStatus(value)

	if err != nil {
		monitor.logs.General.WriteError(logs.ComponentStatus, err.Error())
	} else {
		monitor.logs.General.WriteDebug(logs.ComponentStatus, "completed: %s", monitor.fqdn)
	}
}

// Монитор статуса одного kannel.
func (monitor *Monitor) statusMonitor(ctx context.Context, wg *sync.WaitGroup) {
	monitor.logs.General.WriteDebug(logs.ComponentStatus, "monitor started: %s", monitor.host)

	ticker := time.NewTicker(monitor.config.StatusInterval.Duration)

LOOP:
	for {
		monitor.statusOnce(ctx)

		select {
		case <-ticker.C:
			continue
		case <-ctx.Done():
			break LOOP
		}
	}

	ticker.Stop()

	monitor.logs.General.WriteDebug(logs.ComponentStatus, "monitor stopped: %s", monitor.host)

	wg.Done()
}

// Монитор kannel-ов.
func (service *Discovery) Monitor(ctx context.Context, wg *sync.WaitGroup) {
	for _, entry := range service.kannels {
		wg.Add(1)
		go entry.pingMonitor(ctx, wg)

		wg.Add(1)
		go entry.statusMonitor(ctx, wg)
	}

	wg.Done()
}
