package kannel

import (
	"context"
	"crypto/tls"
	"fmt"
	"io/ioutil"
	"net"
	"net/http"
	"net/url"
	"time"
)

/*
HTTP-клиент. В основном используется для работы с kannel.
*/

// Транспорт по умолчанию для HTTP.
//
// Отправка одной sms на практике занимает до 100 ms.
// Согласно PASSPINCIDENTS-42 на хост может прилететь до 7K+ rpm (~120+ rps).
// Для обработки такого потока нужно держать 120 rps * 0.1 s = 12 активных
// соединений с kannel, плюс одно на ping, одно status и одно на '+' = 15.
var defaultHTTPTransport = &http.Transport{
	MaxIdleConns:          15 * 2,           // (ping + status + sendsms) * кол-во хостов kannel
	MaxIdleConnsPerHost:   15,               // ping + status + sendsms
	IdleConnTimeout:       32 * time.Second, // актуальная настройка nginx
	MaxConnsPerHost:       512,              // актуальные nginx worker_processes * worker_connections
	TLSHandshakeTimeout:   5 * time.Second,
	ExpectContinueTimeout: 1 * time.Second,
	ForceAttemptHTTP2:     true,
	TLSClientConfig: &tls.Config{
		MinVersion:         tls.VersionTLS12,
		ClientSessionCache: tls.NewLRUClientSessionCache(2 * 512),
	},
}

// HTTP-клиент по умолчанию.
var defaultHTTPClient = &http.Client{
	// Таймаут в 5 секунд сильно большой для обычных запросов и маленький,
	// чтобы дать шанс приложению сохранить работоспособность, если где-то
	// забыли установить таймаут в контексте.
	Timeout:   time.Duration(5) * time.Second,
	Transport: defaultHTTPTransport,
}

// Неожиданный код ответа HTTP.
type UnexpectedHTTPCodeError struct {
	actual   int    // Актуальный код.
	expected int    // Ожидаемый код.
	content  []byte // Содержимое ответа.
}

// error interface
func (e *UnexpectedHTTPCodeError) Error() string {
	if len(e.content) > 0 {
		return fmt.Sprintf("unexpected http code - expected: %d, actual: %d, response: '%s'", e.expected, e.actual, string(e.content))
	}

	return fmt.Sprintf("unexpected http code - expected: %d, actual: %d", e.expected, e.actual)
}

// Неожиданное / ошибочное содержимое ответа HTTP.
type UnexpectedHTTPResponseError struct {
	actual   []byte // Актуальный ответ.
	expected []byte // Ожидаемый ответ.
	err      error  // Ошибка.
}

// error interface
func (e *UnexpectedHTTPResponseError) Error() string {
	if e.err == nil {
		return fmt.Sprintf("unexpected http response - expected: '%s', actual: '%s'", string(e.expected), string(e.actual))
	}

	return fmt.Sprintf("unexpected http response - '%s', actual: '%s'", e.err, string(e.actual))
}

// Получение имени хоста из url.
func FqdnFromURL(host string) string {
	u, err := url.Parse(host)
	if err == nil {
		var hp string
		if len(u.Host) != 0 {
			hp = u.Host
		} else {
			hp = host
		}

		h, _, err := net.SplitHostPort(hp)
		if err == nil && len(hp) != 0 {
			host = h
		} else {
			host = hp
		}
	}

	return host
}

// Простой HTTP GET-запрос с клиентом и заголовками по умолчанию.
func httpSimpleGet(ctx context.Context, client *http.Client, query string) (int, []byte, error) {
	request, err := http.NewRequest("GET", query, nil)
	if err != nil {
		return 0, nil, err
	}

	request = request.WithContext(ctx)

	request.Header.Set("User-Agent", "yasms")

	if client == nil {
		client = defaultHTTPClient
	}

	response, err := client.Do(request)
	if err != nil {
		return 0, nil, err
	}

	if response.StatusCode >= http.StatusInternalServerError {
		// при получении 5xx от рила необходимо закрыть соединение, т.к. при
		// работе через l3 балансер следующий запрос может попасть в тот же рил
		response.Close = true
	}

	data, err := ioutil.ReadAll(response.Body)

	_ = response.Body.Close()

	if err != nil {
		return 0, nil, err
	}

	return response.StatusCode, data, nil
}
