package kannel

import (
	"bytes"
	"context"
	"encoding/xml"
	"fmt"
	"math/rand"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	"golang.org/x/text/encoding/unicode"

	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/storage"
)

/*
Работа с HTTP-интерфейсом kannel (пингер, получение статуса, отправка sms).
Документация: https://www.kannel.org/download/1.4.0/userguide-1.4.0/userguide.html.

Перед каждым kannel стоит nginx терминирующий HTTPS и реализующий базовый ACL.
TVM2 пока не работает (см. PASSP-19573).
*/

// Статус гейта kannel (выборочные поля из вывода /status.xml).
type KannelGateStatus struct {
	name   string        // Имя гейта kannel.
	online bool          // Флаг доступности гейта kannel.
	queued uint64        // Размер очереди sms на отправку в гейт kannel.
	seed   int64         // Случайное число для равномерной балансировки.
	kannel *KannelStatus // Родительский kannel (parent).
}

// Статус хоста kannel (выборочные поля из вывода /status.xml).
type KannelStatus struct {
	host   string                       // Имя хоста kannel.
	online bool                         // Флаг доступности kannel.
	queued uint64                       // Размер очереди sms в RAM для kannel.
	stored uint64                       // Размер очереди sms на HDD для kannel.
	gates  map[string]*KannelGateStatus // Статусы гейтов kannel.
}

// Описатель sms сообщения, отправляемого в kannel.
type KannelSms struct {
	User     string // Имя пользователя kannel.
	Password string // Пароль пользователя kannel.
	SmsC     string // Имя гейта kannel.
	From     string // Имя отправителя (альфа-имя).
	Phone    string // Номер телефона получателя.
	Coding   int    // Кодировка текста сообщения (см. константы KANNEL_CODING_*).
	Text     []byte // Текст сообщения.
	Dlr      string // DLR-URL.
	Sender   string // Параметр sender при запросе в kannel (используется в statbox логах на стороне kannel).
	GUID     string // PASSPADMIN-6851, global sms id для логгирования на стороне kannel.
}

// Константы кодировок kannel.
const (
	KannelConding7Bit = 0 // 7-и битная кодировка kannel.
	KannelCodingUCS2  = 2 // UCS-2 (UTF-16-BE) кодировка kannel.
)

// Константы длин сегментов.
const (
	Kannel7BitSegmentSize = 160 // Размер 7-и битного сегмента в байтах.
)

// Ручки на стороне nginx перед kannel.
const (
	KannelPingHandler   = "/ping.html"     // Ручка пингера kannel (touch-файл закрытия от нагрузки).
	KannelStatusHandler = "/kannel-status" // Ручка получения статуса kannel (проксирование в /status.xml).
	KannelSendHandler   = "/sendsms.cgi"   // Ручка отправки sms в kannel (проксирование в /sendsms.cgi).
)

// KannelPingResponse - требуемый ответ на ручку пингера.
var KannelPingResponse = []byte("Pong\n")

// Вызов ручки kannel (проксирование через nginx) и проверка HTTP-кода ответа.
func kannelRequest(ctx context.Context, client *http.Client, query string, expected int) ([]byte, error) {
	code, data, err := httpSimpleGet(ctx, client, query)
	if err != nil {
		// сокрытие url запроса в kannel, т.к. url может содержать секретные данные
		if e, ok := err.(*url.Error); ok {
			return nil, e.Unwrap()
		}

		return nil, err
	}

	if code != expected {
		return nil, &errs.UnexpectedHTTPCodeError{
			Expected: expected,
			Actual:   code,
			Content:  data,
		}
	}

	return data, nil
}

// Вызов ручки пингера kannel.
func kannelPing(ctx context.Context, client *http.Client, host string) (bool, error) {
	data, err := kannelRequest(ctx, client, host+KannelPingHandler, http.StatusOK)
	if err != nil {
		// HTTP-403 при снятии нагрузки - ожидаемый код,
		// логгировать стандартное тело ответа в этом случае лишнее
		if e, ok := err.(*errs.UnexpectedHTTPCodeError); ok && e.Actual == http.StatusForbidden {
			e.Content = []byte("")
			return false, e
		}

		return false, err
	}

	if !bytes.Equal(data, KannelPingResponse) {
		return false, &errs.UnexpectedHTTPResponseError{
			Actual:   data,
			Expected: KannelPingResponse,
		}
	}

	return true, nil
}

// Парсинг статуса kannel.
func parseKannelStatus(data []byte, host string) (*KannelStatus, error) {
	// статус гейта
	type XMLSmsc struct {
		AdminID string `xml:"admin-id"`
		Status  string `xml:"status"`
		Queued  uint64 `xml:"queued"`
	}

	// статус kannel
	type XMLKannel struct {
		XMLName   xml.Name   `xml:"gateway"`
		Status    string     `xml:"status"`
		Queued    uint64     `xml:"sms>sent>queued"`
		StoreSize uint64     `xml:"sms>storesize"`
		SMSCs     []*XMLSmsc `xml:"smscs>smsc"`
	}

	var result XMLKannel
	err := xml.Unmarshal(data, &result)
	if err != nil {
		return nil, &errs.UnexpectedHTTPResponseError{
			Actual:  data,
			Message: err,
		}
	}

	kannel := &KannelStatus{
		host:   host,
		online: strings.HasPrefix(result.Status, "running"),
		queued: result.Queued,
		stored: result.StoreSize,
		gates:  make(map[string]*KannelGateStatus),
	}

	for _, smsc := range result.SMSCs {
		kannel.gates[smsc.AdminID] = &KannelGateStatus{
			name:   smsc.AdminID,
			queued: smsc.Queued,
			online: strings.HasPrefix(smsc.Status, "online"),
			seed:   rand.Int63(),
			kannel: kannel,
		}
	}

	return kannel, nil
}

// Вызов ручки получения статуса kannel.
func kannelStatus(ctx context.Context, client *http.Client, host string) (*KannelStatus, error) {
	data, err := kannelRequest(ctx, client, host+KannelStatusHandler, http.StatusOK)
	if err != nil {
		return nil, err
	}

	return parseKannelStatus(data, host)
}

// Определение кодировки текста для отправки в kannel.
func kannelCoding(data []byte) int {
	// PASSP-8025 - длинные сообщения в 7-и битной кодировке не проходят
	// TODO: имеет смысл выяснить актуальность данного утверждения
	if len(data) > Kannel7BitSegmentSize {
		return KannelCodingUCS2
	}

	// В базу сообщения пишутся as is (предполагается utf-8, но это никак не
	// проверяется). В kannel можно отправить сообщение в 7-и битной ascii
	// если оно может быть перекодировано в gsm 03.38 и в ucs-2 в противном
	// случае. При этом часть ascii не может быть закодирована gsm.
	for _, c := range data {
		if (c >= 0x20 /* <SPACE> */ && c < 0x7f /* <DEL> */ && c != 0x60 /* GRAVE ACCENT "`" */) ||
			c == 0x0a /* <LF> */ ||
			c == 0x0d /* <CR> */ ||
			c == 0x0c /* <FF> */ {
			continue
		}

		return KannelCodingUCS2
	}

	return KannelConding7Bit
}

// Кодирование текста sms и установка параметра coding.
func KannelEncodeSms(sms *KannelSms) error {
	sms.Coding = kannelCoding(sms.Text)

	if sms.Coding == KannelCodingUCS2 {
		encoder := unicode.UTF16(unicode.BigEndian, unicode.IgnoreBOM).NewEncoder()
		utf16, err := encoder.Bytes(sms.Text)
		if err != nil {
			return err
		}

		sms.Text = utf16
	}

	return nil
}

// Формирование DLR-URL на основании имени хоста, global и local sms id.
func KannelDlrURL(host, guid string, smsid uint64) string {
	return host + "?" +
		"gmsgid=" + guid + "&" +
		"msgid=" + strconv.FormatUint(smsid, 10) + "&" +
		"gate=%i&from=%p&to=%P&id=%I&status=%d&answer=%A&ts=%t&meta=%D"
}

// Преобразование sms в строку логов.
func (sms *KannelSms) String() string {
	return fmt.Sprintf(
		"guid=%s, phone=%s, smsc=%s, from=%s, coding=%d, sender=%s, text=[%d], dlr=[%d]",
		sms.GUID,
		storage.MaskedPhone(sms.Phone),
		sms.SmsC,
		sms.From,
		sms.Coding,
		sms.Sender,
		len(sms.Text),
		len(sms.Dlr),
	)
}

// Вызов ручки отправки sms.
func KannelSendSms(ctx context.Context, client *http.Client, host string, sms *KannelSms) error {
	params := url.Values{}

	// PASSPADMIN-6851, проброс global sms id в логи kannel
	// В коде kannel 1.4.4-4yandex21 для этих целей специально
	// отключена замена submit_sm.service_type на значение binfo
	if len(sms.GUID) != 0 {
		params.Add("binfo", sms.GUID)
	}

	// TODO: возможно username/password имеет смысл перенести в заголовки,
	// чтобы они не светились в логах
	params.Add("smsc", sms.SmsC)
	params.Add("username", sms.User)
	params.Add("password", sms.Password)
	params.Add("from", sms.From)
	params.Add("to", sms.Phone)
	params.Add("coding", strconv.FormatUint(uint64(sms.Coding), 10))
	params.Add("text", string(sms.Text))

	if len(sms.Sender) != 0 {
		params.Add("sender", sms.Sender)
	}

	if len(sms.Dlr) != 0 {
		params.Add("dlr-mask", "31")
		params.Add("dlr-url", sms.Dlr)
	}

	_, err := kannelRequest(ctx, client, host+KannelSendHandler+"?"+params.Encode(), http.StatusAccepted)
	if err != nil {
		return err
	}

	return nil
}
