package logs

import (
	"fmt"
	"os"
	"sync"
	"time"
)

/*
Базовая работа с логами.
*/

// Имена компонент для логов.
const (
	ComponentSignal    = "signal"    // Имя компоненты мониторинга сигналов.
	ComponentPing      = "ping"      // Имя компоненты для пингера kannel.
	ComponentStatus    = "status"    // Имя компоненты для получения статуса kannel.
	ComponentYasmsInt  = "yasmsint"  // Имя компоненты для получения списка гейтов и фоллбеков из yasms internal.
	ComponentQueue     = "queue"     // Имя компоненты для получения списка sms в очереди.
	ComponentHeartbeat = "beat"      // Имя компоненты для обновления статуса живости демона.
	ComponentSms       = "sms"       // Имя компоненты для работы с sms.
	ComponentKannel    = "kannel"    // Имя компоненты для работы с kannel.
	ComponentKeyring   = "keyring"   // Имя компоненты для работы с keyring.
	ComponentDecryptor = "decryptor" // Имя компоненты для расшифровки сообщений.
	ComponentAntiFraud = "antifraud" // Имя компоненты для антифрода.
)

// Базовый лог.
type GeneralLog struct {
	sync.Mutex
	name string   // Имя файла.
	fd   *os.File // Дескриптор файла.
}

// Создание базового лога.
func NewGeneralLog(name string) *GeneralLog {
	return &GeneralLog{
		name: name,
	}
}

// Закрытие базового лога.
func (log *GeneralLog) Close() {
	if log.fd != nil {
		_ = log.fd.Close()
		log.fd = nil
	}
}

// Открытие базового лога.
func (log *GeneralLog) Open() error {
	log.Close()

	if len(log.name) != 0 {
		var err error
		log.fd, err = os.OpenFile(log.name, os.O_CREATE|os.O_APPEND|os.O_WRONLY, 0644)
		if err != nil {
			return err
		}
	}

	return nil
}

// Переоткрытие базового лога.
func (log *GeneralLog) ReOpen() error {
	log.Lock()
	defer log.Unlock()

	return log.Open()
}

// Запись в базовый лог.
func (log *GeneralLog) Write(format string, a ...interface{}) {
	log.Lock()
	defer log.Unlock()

	if log.fd != nil {
		if len(a) != 0 {
			_, _ = fmt.Fprintf(log.fd, format, a...)
		} else {
			_, _ = log.fd.WriteString(format)
		}
	}
}

// Стандартная запись в базовый лог.
func (log *GeneralLog) WriteGeneral(component, severity, format string, a ...interface{}) {
	if len(a) != 0 {
		format = fmt.Sprintf(format, a...)
	}

	log.Write(time.Now().Format("2006-01-02 15:04:05.000") + " " +
		severity + " " +
		component + " " +
		format + "\n",
	)
}

// Запись в базовый лог отладочной информации.
func (log *GeneralLog) WriteDebug(component, format string, a ...interface{}) {
	log.WriteGeneral(component, "DEBUG", format, a...)
}

// Запись в базовый лог отладочной информации.
func (log *GeneralLog) WriteWarning(component, format string, a ...interface{}) {
	log.WriteGeneral(component, "WARNING", format, a...)
}

// Запись в базовый лог информации об ошибке.
func (log *GeneralLog) WriteError(component, format string, a ...interface{}) {
	log.WriteGeneral(component, "ERROR", format, a...)
}
