package logs

import (
	"net"
	"strconv"
	"time"

	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/errs"
	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/fraud"
)

/*
Работа с графитными (мониторинговыми) логами.
*/

// Имена сервисов для графитных логов.
const (
	ServiceKannel    = "kannel"    // Имя сервиса для kannel.
	ServiceAntifraud = "antifraud" // Имя сервиса для антифрода.
	ServiceStorage   = "db"        // Имя сервиса для хранилища.
	ServiceYasmsInt  = "yasmsint"  // Имя сервиса для походов в yasms internal.
)

// Имена действий для графитных логов.
const (
	ActionKannelPing     = "ping"      // Имя действия пингера kannel.
	ActionKannelStatus   = "status"    // Имя действия получения статуса kannel.
	ActionKannelSend     = "send"      // Имя действия для отправки sms kannel.
	ActionAntifraudCheck = "check"     // Имя действия для проверки sms в антифроде.
	ActionGatesList      = "gates"     // Имя действия получения списка гейтов из хранилища.
	ActionFallbacksList  = "fallbacks" // Имя действия получения списка фоллбеков из хранилища.
	ActionQueueList      = "queue"     // Имя действия получения списка sms в очереди.
	ActionHeartbeat      = "beat"      // Имя действия обновления статуса живости демона.
	ActionSmsStatus      = "status"    // Имя действия обновления статуса sms.
)

// Графитный лог.
type GraphiteLog struct {
	log *StatboxLog
}

// Запись в графитном логе.
type GraphiteRecord struct {
	graphite *GraphiteLog // Лог файл, к которому относится запись.
	started  time.Time    // Время начала события.
	hostname string       // Хост сервиса (обычно fqdn без схемы).
	service  string       // Имя сервиса.
	action   string       // Логгируемое действие.
	retries  uint64       // Количество оставшихся попыток.
	code     int          // HTTP код ответа.
}

// Возможные варианты поля response.
const (
	ResponseSuccess = "success"
	ResponseTimeout = "timeout"
	ResponseFailed  = "failed"
)

// Создание нового графитного лога.
func NewGraphiteLog(name string) *GraphiteLog {
	return &GraphiteLog{
		log: NewStatboxLog(name),
	}
}

// GeneralLog::close()
func (graphite *GraphiteLog) Close() {
	graphite.log.Close()
}

// GeneralLog::reopen()
func (graphite *GraphiteLog) ReOpen() error {
	return graphite.log.ReOpen()
}

// Создание новой графитной записи и отсечка времени начала.
func (graphite *GraphiteLog) NewRecord(hostname, service, action string, retries uint64) *GraphiteRecord {
	return &GraphiteRecord{
		graphite: graphite,
		started:  time.Now(),
		hostname: hostname,
		service:  service,
		action:   action,
		retries:  retries,
	}
}

// Создание новой графитной HTTP-записи и отсечка времени начала.
func (graphite *GraphiteLog) NewHTTPRecord(hostname, service, action string, expected int, retries uint64) *GraphiteRecord {
	rec := graphite.NewRecord(hostname, service, action, retries)
	rec.code = expected
	return rec
}

// Запись графитной http-записи в лог.
func (rec *GraphiteRecord) Close(err error) {
	var response string
	if err == nil {
		response = ResponseSuccess
	} else {
		switch e := err.(type) {
		case net.Error:
			rec.code = 0
			if e.Timeout() {
				response = ResponseTimeout
			} else {
				response = ResponseFailed
			}
		case *errs.UnexpectedHTTPCodeError:
			rec.code = e.Actual
			response = ResponseFailed
		case *fraud.UnexpectedHTTPCodeError:
			rec.code = e.ActualStatusCode
			response = ResponseFailed
		case *errs.UnexpectedHTTPResponseError:
			response = ResponseFailed
		default:
			response = ResponseFailed
		}
	}

	if rec.graphite != nil {
		// если в логе использовать значение rec.started, в поле timestamp,
		// то получится немонотонная последовательность в зависимости от duration
		t := time.Now()

		rec.graphite.log.AddSlice([]string{
			"timestamp", t.Format("2006-01-02T15:04:05"),
			"unixtime", strconv.FormatInt(t.Unix(), 10),
			"hostname", rec.hostname,
			"service", rec.service,
			"action", rec.action,
			"response", response,
			"duration", strconv.FormatFloat(time.Since(rec.started).Seconds(), 'f', -1, 64),
			"http_code", strconv.FormatInt(int64(rec.code), 10),
			"retries_left", strconv.FormatUint(rec.retries, 10),
		}, nil)

		rec.graphite = nil
	}
}
