package logs

import (
	"errors"
)

/*
Группировка стандартного набора логов в одну структуру для упрощения работы.
*/

type Config struct {
	Common         string `json:"common"`
	Graphite       string `json:"graphite"`
	StatboxPublic  string `json:"statbox_public"`
	StatboxPrivate string `json:"statbox_private"`
}

// Стандартный набор логов.
type Logs struct {
	General  *GeneralLog        // Базовый лог.
	Graphite *GraphiteLog       // Графитный лог.
	Public   *PublicStatboxLog  // Публичный statbox лог.
	Private  *PrivateStatboxLog // Приватный statbox лог.
}

// Создание и открытие набора логов.
func NewLogs(config *Config) *Logs {
	return &Logs{
		General:  NewGeneralLog(config.Common),
		Graphite: NewGraphiteLog(config.Graphite),
		Public:   NewPublicStatboxLog(config.StatboxPublic),
		Private:  NewPrivateStatboxLog(config.StatboxPrivate),
	}
}

func NewNullLogs() *Logs {
	devnull := "/dev/null"
	return NewLogs(&Config{
		Common:         devnull,
		Graphite:       devnull,
		StatboxPublic:  devnull,
		StatboxPrivate: devnull,
	})
}

// Закрытие набора логов.
func (logs *Logs) Close() {
	logs.General.Close()
	logs.Graphite.Close()
	logs.Public.Close()
	logs.Private.Close()
}

// Переоткрытие набора логов.
func (logs *Logs) ReOpen() error {
	var result string

	if err := logs.General.ReOpen(); err != nil {
		result = err.Error() + ". "
	}

	if err := logs.Graphite.ReOpen(); err != nil {
		result = result + err.Error() + ". "
	}

	if err := logs.Public.ReOpen(); err != nil {
		result = result + err.Error() + ". "
	}

	if err := logs.Private.ReOpen(); err != nil {
		result = result + err.Error() + ". "
	}

	if result != "" {
		return errors.New(result)
	}

	return nil
}
