package logs

import (
	"strings"
)

/*
Работа с логами статистики.

Известные обработчики статистики:

- https://github.yandex-team.ru/statbox/outer-job/blob/master/passp-12641/main.py
- https://github.yandex-team.ru/statbox/outer-job/blob/master/passp-16535/main.py
- https://github.yandex-team.ru/statbox/outer-job/blob/master/passp-16535-2/main.py
*/

// Стандартные имена полей.
const (
	StatboxFieldNumber = "number" // Имя поля с немаскированным номером в statbox логе.
	StatboxFieldPrice  = "price"  // Имя поля с ценой sms сообщения в statbox логе.
)

var escapeChars = map[byte]string{
	'\t':   "\\t",
	'\n':   "\\n",
	'\r':   "\\r",
	'\\':   "\\\\",
	'\x00': "\\0",
	'=':    "\\=",
	'"':    "\\\"",
}

func escape(raw string) string {
	escaped := make([]string, 0, len(raw))
	for i := range raw {
		val, ok := escapeChars[raw[i]]
		if ok {
			escaped = append(escaped, val)
		} else {
			escaped = append(escaped, string(raw[i]))
		}
	}
	return strings.Join(escaped, "")
}

// statbox лог.
type StatboxLog struct {
	log *GeneralLog
}

// Создание нового statbox лога.
func NewStatboxLog(name string) *StatboxLog {
	return &StatboxLog{
		log: NewGeneralLog(name),
	}
}

// statbox лог.
type PublicStatboxLog struct {
	StatboxLog
}

func NewPublicStatboxLog(name string) *PublicStatboxLog {
	return &PublicStatboxLog{
		StatboxLog: StatboxLog{log: NewGeneralLog(name)},
	}
}

type PrivateStatboxLog struct {
	StatboxLog
}

func NewPrivateStatboxLog(name string) *PrivateStatboxLog {
	return &PrivateStatboxLog{
		StatboxLog: StatboxLog{log: NewGeneralLog(name)},
	}
}

// GeneralLog::close()
func (statbox *StatboxLog) Close() {
	statbox.log.Close()
}

// GeneralLog::reopen()
func (statbox *StatboxLog) ReOpen() error {
	return statbox.log.ReOpen()
}

// Добавление записи в statbox лог.
//
// Длина входящих параметров должна быть кратна 2 (key = value),
// проверок выхода за границы не производится (будет panic).
func (statbox *StatboxLog) AddSlice(record []string, extra []string) {
	var buf strings.Builder
	buf.Grow(1024)

	buf.WriteString("tskv")

	for i := 0; i < len(record); i += 2 {
		buf.WriteString("\t")
		buf.WriteString(escape(record[i]))
		buf.WriteString("=")
		buf.WriteString(escape(record[i+1]))
	}

	for i := 0; i < len(extra); i += 2 {
		buf.WriteString("\t")
		buf.WriteString(escape(extra[i]))
		buf.WriteString("=")
		buf.WriteString(escape(extra[i+1]))
	}

	buf.WriteString("\n")

	statbox.log.Write(buf.String())
}

// forward declaration to avoid import-loop
type RowLogWriter interface {
	StatboxPrivateSlice(action string, extra []string) []string
	StatboxPublicSlice(action string, extra []string) []string
}

func (statbox *PrivateStatboxLog) AddRow(row RowLogWriter, action string, extraSlice []string, extraRow []string) {
	statbox.AddSlice(row.StatboxPrivateSlice(action, extraSlice), extraRow)
}

func (statbox *PublicStatboxLog) AddRow(row RowLogWriter, action string, extraSlice []string, extraRow []string) {
	statbox.AddSlice(row.StatboxPublicSlice(action, extraSlice), extraRow)
}
