package routing

import (
	"fmt"

	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/config"
	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/kannel"
	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/logs"
)

/*
Маршрутизатор sms из хранилища в параметры вызова kannel-api.
*/

// Описатель маршрута для отправки sms.
type RouteEntry struct {
	Host    string     // Имя хоста kannel.
	SmsC    string     // Имя гейта kannel.
	From    string     // Имя отправителя.
	Dlr     string     // DLR-URL.
	Initial *GateEntry // Изначальный гейт в случае fallback.
}

// Маршрутизатор sms.
type Router struct {
	gates     GatesProvider     // Информация о Гейтах
	fallbacks FallbacksProvider // Информация о Фоллбеках
	kannels   *kannel.Discovery // Kannel service discovery.
	dlr       string            // DLR-URL
	logs      *logs.Logs        // Логи.
}

// Временная ошибка маршрутизации.
type RouteTemporaryError struct{}

// error interface
func (e *RouteTemporaryError) Error() string {
	return "route temporary error"
}

// Создание маршрутизатора sms из хранилища в параметры вызова kannel-api.
func NewRouter(gates GatesProvider, fallbacks FallbacksProvider, kannels *kannel.Discovery, dlr string, loggers *logs.Logs) *Router {
	return &Router{
		gates:     gates,
		fallbacks: fallbacks,
		kannels:   kannels,
		dlr:       dlr,
		logs:      loggers,
	}
}

// Преобразование маршрута в строку логов.
func (entry *RouteEntry) String() string {
	return fmt.Sprintf(
		"host=%s, smsc=%s, from=%s, dlr=[%d]",
		kannel.FqdnFromURL(entry.Host),
		entry.SmsC,
		entry.From,
		len(entry.Dlr),
	)
}

// Получение маршрута по id гейта
// smsid и guid используются для формирования dlr
func (router *Router) Entry(gateid, smsid uint64, guid string) (*RouteEntry, error) {
	gate := router.gates.GateEntry(gateid)
	if gate == nil {
		return nil, &RouteTemporaryError{}
	}

	// стандартное обнаружение
	host := router.kannels.Discovery(gate.SmsC)
	if len(host) != 0 {
		return &RouteEntry{
			Host: host,
			SmsC: gate.SmsC,
			From: gate.From,
			Dlr:  kannel.KannelDlrURL(router.dlr, guid, smsid),
		}, nil
	}

	// поиск fallback маршрута
	host, fallback := router.Fallback(gate.From, gate.SmsC)
	if fallback != nil {
		return &RouteEntry{
			Host:    host,
			SmsC:    fallback.SmsC,
			From:    fallback.From,
			Dlr:     kannel.KannelDlrURL(router.dlr, guid, smsid),
			Initial: gate,
		}, nil
	}

	// гейты для фейковой отправки.
	switch gate.SmsC {
	case config.GateNull:
		// в statbox лог ничего не пишется (аналог письма в спортлото)
		return &RouteEntry{
			SmsC: gate.SmsC,
			From: gate.From,
		}, nil
	case config.GateDelivered:
		// в statbox лог пишется информация об успешной отправке в гейт
		// и получения dlr (подтверждения доставки абоненту
		return &RouteEntry{
			SmsC: gate.SmsC,
			From: gate.From,
		}, nil
	}

	// ничего не найдено
	return nil, &RouteTemporaryError{}
}

// Поиск fallback хоста kannel по имени отправителя и гейта.
func (router *Router) Fallback(from, smsc string) (string, *FallbackEntry) {
	for _, entry := range router.fallbacks.Fallbacks(from, smsc) {
		host := router.kannels.Discovery(entry.SmsC)
		if len(host) != 0 {
			return host, entry
		}
	}

	return "", nil
}
