package yasmsint

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"time"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/shared/golibs/utils"
)

const TvmYasmsInternalAlias string = "yasms_internal"
const HeaderServiceTicket string = "X-Ya-Service-Ticket"

type ClientConfig struct {
	Host string `json:"host"`
	Port uint16 `json:"port"`

	RequestTimeout utils.Duration `json:"request_timeout"`
}

type RawGateEntry struct {
	ID   string `json:"gateid"`
	SmsC string `json:"aliase"`
	From string `json:"fromname"`
}

type GetGatesResponse struct {
	Gates   []*RawGateEntry `json:"gates"`
	NextURL string          `json:"next"`
}

type RawFallbackEntry struct {
	SrcGate string `json:"srcgate"`
	SrcName string `json:"srcname"`
	DstGate string `json:"dstgate"`
	DstName string `json:"dstname"`
	Order   int16  `json:"order"`
}

type GetFallbackResponse struct {
	Fallbacks []*RawFallbackEntry `json:"fallbacks"`
	NextURL   string              `json:"next"`
}

type Client struct {
	tvmClient      tvm.Client
	httpClient     *resty.Client
	requestTimeout time.Duration
}

func NewYasmsInternalClient(
	config *ClientConfig,
	tvmClient tvm.Client) *Client {
	httpClient := resty.New()
	httpClient.SetBaseURL(fmt.Sprintf("%s:%d", config.Host, config.Port))

	return &Client{
		tvmClient:      tvmClient,
		httpClient:     httpClient,
		requestTimeout: config.RequestTimeout.Duration,
	}
}

func (client *Client) GetGatesPageURL(min, limit uint64) string {
	return fmt.Sprintf("/1/service/gates?min=%d&limit=%d", min, limit)
}

func (client *Client) GetGatesNextPage(ctx context.Context, nextURL string) (*GetGatesResponse, error) {
	httpResponse, err := client.Get(ctx, nextURL)
	if err != nil {
		return nil, err
	}

	response := &GetGatesResponse{}
	err = json.Unmarshal(httpResponse.Body(), response)
	if err != nil {
		return nil, fmt.Errorf(
			"invalid response body: [%d] %s. %s",
			httpResponse.StatusCode(),
			string(httpResponse.Body()),
			err.Error())
	}

	return response, nil
}

func (client *Client) GetFallbacksPageURL(min, limit uint64) string {
	return fmt.Sprintf("/1/service/fallbacks?min=%d&limit=%d", min, limit)
}

func (client *Client) GetFallbacksNextPage(ctx context.Context, nextURL string) (*GetFallbackResponse, error) {
	httpResponse, err := client.Get(ctx, nextURL)
	if err != nil {
		return nil, err
	}

	response := &GetFallbackResponse{}
	err = json.Unmarshal(httpResponse.Body(), response)
	if err != nil {
		return nil, fmt.Errorf(
			"invalid response body: [%d] %s. %s",
			httpResponse.StatusCode(),
			string(httpResponse.Body()),
			err.Error())
	}

	return response, nil
}

func (client *Client) Get(ctx context.Context, url string) (*resty.Response, error) {
	ticket, err := client.tvmClient.GetServiceTicketForAlias(context.Background(), TvmYasmsInternalAlias)
	if err != nil {
		return nil, fmt.Errorf("failed to fetch service ticket: %s", err.Error())
	}

	ctx, cancel := context.WithTimeout(ctx, client.requestTimeout)
	defer cancel()

	httpResponse, err := client.httpClient.R().
		SetContext(ctx).
		SetHeader(HeaderServiceTicket, ticket).
		Get(url)

	if err != nil {
		return nil, err
	}

	if httpResponse.StatusCode() != http.StatusOK {
		return nil, fmt.Errorf(
			"unexpected http code error: expected %d, got %d. URL: %s. Message: %s",
			http.StatusOK,
			httpResponse.StatusCode(),
			url,
			string(httpResponse.Body()))
	}

	return httpResponse, nil
}
