package yasmsint

import (
	"context"
	"fmt"
	"net/http"
	"time"

	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/logs"
	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/routing"
)

type FallbacksList map[string]map[string][]*routing.FallbackEntry

func (fetcher *Fetcher) SetFallbacks(fallbacks FallbacksList) {
	fetcher.Lock()
	defer fetcher.Unlock()

	fetcher.fallbacks = fallbacks
	fetcher.fallbacksLastUpdateTime = time.Now()
}

func (fetcher *Fetcher) Fallbacks(from, smsc string) []*routing.FallbackEntry {
	fetcher.RLock()
	defer fetcher.RUnlock()

	gates, exists := fetcher.fallbacks[from]
	if !exists {
		return nil
	}

	fallbacks, exists := gates[smsc]
	if !exists {
		return nil
	}

	return fallbacks
}

func (fetcher *Fetcher) FetchFallbacks(ctx context.Context) (FallbacksList, error) {
	fallbacks := newFallbacksCollector()

	nextURL := fetcher.client.GetFallbacksPageURL(0, fetcher.config.FallbacksLimitPerRequest)
	for nextURL != "" {
		record := fetcher.logs.Graphite.NewHTTPRecord(fetcher.config.Client.Host, logs.ServiceYasmsInt, logs.ActionFallbacksList, http.StatusOK, 0)
		response, err := fetcher.client.GetFallbacksNextPage(ctx, nextURL)
		record.Close(err)

		if err != nil {
			return nil, fmt.Errorf("failed to fetch fallbacks: %s", err.Error())
		}

		for _, raw := range response.Fallbacks {
			fallbacks.Append(raw)
		}

		nextURL = response.NextURL
	}

	return fallbacks.Get(), nil
}

func (fetcher *Fetcher) ReloadFallbacks(ctx context.Context) error {
	fetcher.logs.General.WriteDebug(logs.ComponentYasmsInt, "started for fallbacks")

	fallbacks, err := fetcher.FetchFallbacks(ctx)
	if err != nil {
		fetcher.logs.General.WriteError(logs.ComponentYasmsInt, err.Error())
		return err
	}

	fetcher.SetFallbacks(fallbacks)

	fetcher.logs.General.WriteDebug(logs.ComponentYasmsInt, "completed for fallbacks")

	return nil
}
