package yasmsint

import (
	"context"
	"sync"
	"time"

	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/logs"
	"a.yandex-team.ru/passport/shared/golibs/juggler"
	"a.yandex-team.ru/passport/shared/golibs/utils"
)

type FetcherConfig struct {
	Client *ClientConfig `json:"client"`

	UpdateInterval         utils.Duration `json:"update_interval"`
	UpdateWarningThreshold utils.Duration `json:"update_warning_threshold"`

	GatesLimitPerRequest     uint64 `json:"gates_limit_per_request"`
	FallbacksLimitPerRequest uint64 `json:"fallbacks_limit_per_request"`
}

type Fetcher struct {
	sync.RWMutex

	gates     GatesList
	fallbacks FallbacksList

	gatesLastUpdateTime     time.Time
	fallbacksLastUpdateTime time.Time

	client *Client
	logs   *logs.Logs
	config *FetcherConfig
}

func NewYasmsInternalFetcher(
	config *FetcherConfig,
	tvmClient tvm.Client,
	logs *logs.Logs) *Fetcher {
	return &Fetcher{
		gates:  make(GatesList),
		client: NewYasmsInternalClient(config.Client, tvmClient),
		logs:   logs,
		config: config,
	}
}

func (fetcher *Fetcher) HasData() bool {
	return !fetcher.gatesLastUpdateTime.IsZero() && !fetcher.fallbacksLastUpdateTime.IsZero()
}

func (fetcher *Fetcher) GetJugglerStatus() *juggler.Status {
	now := time.Now()

	if !fetcher.HasData() {
		return juggler.NewStatus(
			juggler.Critical,
			"yasms-internal fetcher: no data",
		)
	}

	if now.Sub(fetcher.gatesLastUpdateTime) > fetcher.config.UpdateWarningThreshold.Duration {
		return juggler.NewStatus(
			juggler.Warning,
			"yasms-internal fetcher: gates are outdated for %v", now.Sub(fetcher.gatesLastUpdateTime),
		)
	}

	if now.Sub(fetcher.fallbacksLastUpdateTime) > fetcher.config.UpdateWarningThreshold.Duration {
		return juggler.NewStatus(
			juggler.Warning,
			"yasms-internal fetcher: fallbacks are outdated for %v", now.Sub(fetcher.fallbacksLastUpdateTime),
		)
	}

	return juggler.NewStatusOk()
}

func (fetcher *Fetcher) Reload(ctx context.Context) error {
	if err := fetcher.ReloadGates(ctx); err != nil {
		return err
	}
	if err := fetcher.ReloadFallbacks(ctx); err != nil {
		return err
	}
	return nil
}

func (fetcher *Fetcher) Monitor(ctx context.Context, wg *sync.WaitGroup) {
	fetcher.logs.General.WriteDebug(logs.ComponentYasmsInt, "monitor started for %s", fetcher.config.Client.Host)

	ticker := time.NewTicker(fetcher.config.UpdateInterval.Duration)

LOOP:
	for {
		_ = fetcher.Reload(ctx)

		select {
		case <-ticker.C:
			continue
		case <-ctx.Done():
			break LOOP
		}
	}

	ticker.Stop()

	fetcher.logs.General.WriteDebug(logs.ComponentYasmsInt, "monitor stopped for %s", fetcher.config.Client.Host)

	wg.Done()
}
