package yasmsint

import (
	"context"
	"fmt"
	"net/http"
	"strconv"
	"time"

	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/logs"
	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/routing"
)

type GatesList map[uint64]*routing.GateEntry

func (fetcher *Fetcher) SetGates(gates GatesList) {
	fetcher.Lock()
	defer fetcher.Unlock()

	fetcher.gates = gates
	fetcher.gatesLastUpdateTime = time.Now()
}

func (fetcher *Fetcher) GateEntry(id uint64) *routing.GateEntry {
	fetcher.RLock()
	defer fetcher.RUnlock()

	gate, exists := fetcher.gates[id]
	if !exists {
		return nil
	}

	return gate
}

func (fetcher *Fetcher) FetchGates(ctx context.Context) (GatesList, error) {
	gates := make(GatesList)

	nextURL := fetcher.client.GetGatesPageURL(0, fetcher.config.GatesLimitPerRequest)
	for nextURL != "" {
		record := fetcher.logs.Graphite.NewHTTPRecord(fetcher.config.Client.Host, logs.ServiceYasmsInt, logs.ActionGatesList, http.StatusOK, 0)
		response, err := fetcher.client.GetGatesNextPage(ctx, nextURL)
		record.Close(err)

		if err != nil {
			return nil, fmt.Errorf("failed to fetch gates: %s", err.Error())
		}

		for _, raw := range response.Gates {
			id, err := strconv.ParseUint(raw.ID, 10, 64)
			if err != nil {
				return nil, fmt.Errorf("failed to parse gate id: %s. %s", raw.ID, err.Error())
			}

			gates[id] = &routing.GateEntry{
				ID:   id,
				SmsC: raw.SmsC,
				From: raw.From,
			}
		}

		nextURL = response.NextURL
	}

	return gates, nil
}

func (fetcher *Fetcher) ReloadGates(ctx context.Context) error {
	fetcher.logs.General.WriteDebug(logs.ComponentYasmsInt, "started for gates")

	gates, err := fetcher.FetchGates(ctx)
	if err != nil {
		fetcher.logs.General.WriteError(logs.ComponentYasmsInt, err.Error())
		return err
	}

	fetcher.SetGates(gates)

	fetcher.logs.General.WriteDebug(logs.ComponentYasmsInt, "completed for gates")

	return nil
}
