package storage

import (
	"context"
	"sync"
	"time"

	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/logs"
	"a.yandex-team.ru/passport/shared/golibs/utils"
)

/*
Обновление информации о живости демона в хранилище.

Процессы API периодически проверяют timestamp в базе с именем своего хоста
в качестве ключа и, если timestamp сильно отличается, закрываются от нагрузки
(ping от балансера) предполагая, что демон не работает (перегружен / упал).
*/

type HeartbeatConfig struct {
	Interval utils.Duration `json:"interval"`
	Timeout  utils.Duration `json:"timeout"`
	Fqdn     string         `json:"fqdn"`
}

// Обновлятор в хранилище статуса живости демона.
type Heartbeat struct {
	sync.RWMutex
	storage *Storage   // Хранилище.
	logs    *logs.Logs // Логи.
	config  *HeartbeatConfig
}

// Создание нового обновлятора живости демона.
func NewHeartbeat(storage *Storage, config *HeartbeatConfig, logs *logs.Logs) *Heartbeat {
	return &Heartbeat{
		storage: storage,
		logs:    logs,
		config:  config,
	}
}

// Разовое выполнение обновления живости демона.
func (heartbeat *Heartbeat) do(ctx context.Context) {
	heartbeat.logs.General.WriteDebug(logs.ComponentHeartbeat, "started")

	timeout, cancel := context.WithTimeout(ctx, heartbeat.config.Timeout.Duration)
	defer cancel()

	record := heartbeat.logs.Graphite.NewRecord(heartbeat.storage.Host, logs.ServiceStorage, logs.ActionHeartbeat, 0)
	err := execHeartbeat(timeout, heartbeat.storage.DB, heartbeat.config.Fqdn)
	record.Close(err)

	if err != nil {
		heartbeat.logs.General.WriteError(logs.ComponentHeartbeat, err.Error())
	} else {
		heartbeat.logs.General.WriteDebug(logs.ComponentHeartbeat, "completed")
	}
}

// Периодическое выполнение обновления живости демона.
func (heartbeat *Heartbeat) Monitor(ctx context.Context, wg *sync.WaitGroup) {
	heartbeat.logs.General.WriteDebug(logs.ComponentHeartbeat, "monitor started for %s", heartbeat.storage.URL)

	ticker := time.NewTicker(heartbeat.config.Interval.Duration)

LOOP:
	for {
		heartbeat.do(ctx)

		select {
		case <-ticker.C:
			continue
		case <-ctx.Done():
			break LOOP
		}
	}

	ticker.Stop()

	heartbeat.logs.General.WriteDebug(logs.ComponentHeartbeat, "monitor stopped for %s", heartbeat.storage.URL)

	wg.Done()
}
